/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.http.protocol;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.services.security.PasswordUtil;
import org.jboss.soa.esb.util.ssl.SSLUtil;

import java.io.File;
import java.io.InputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.KeyStoreException;
import java.security.cert.CertificateException;
import java.util.Properties;

/**
 * Abstract {@link ProtocolSocketFactoryBuilder} providing utility methods
 * for concrete implementations.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class AbstractProtocolSocketFactoryBuilder implements ProtocolSocketFactoryBuilder {

    private Properties configuration;

    public void setConfiguration(Properties configuration) throws ConfigurationException {
        this.configuration = configuration;
    }

    protected String getTruststorePassword() {
        return configuration.getProperty("truststore-passw");
    }

    protected String getKeystorePassword() {
        return configuration.getProperty("keystore-passw");
    }

    protected String getKeystoreType() {
        return configuration.getProperty("keystore-type", "jks");
    }

    protected String getTruststoreType() {
        return configuration.getProperty("truststore-type", "jks");
    }

    protected URL getTruststoreURL() throws ConfigurationException {
        URL trustStoreURL = null;
        String trustStoreConfig = configuration.getProperty("truststore");

        if(trustStoreConfig != null) {
            try {
                trustStoreURL = new URL(trustStoreConfig);
            } catch (MalformedURLException e) {
                final URL fileURL = getFileAsURL(trustStoreConfig) ;
                if (fileURL != null)
                {
                    return fileURL ;
                }
                throw new ConfigurationException("Invalid 'truststore' config.  Must be valid URL.", e);
            }
        }

        return trustStoreURL;
    }

    protected URL getKeystoreURL() throws ConfigurationException {
        String keyStoreConfig = configuration.getProperty("keystore");
        URL keyStoreURL = null;

        if(keyStoreConfig != null) {
            try {
                keyStoreURL = new URL(keyStoreConfig);
            } catch (MalformedURLException e) {
                final URL fileURL = getFileAsURL(keyStoreConfig) ;
                if (fileURL != null)
                {
                    return fileURL ;
                }
                throw new ConfigurationException("Invalid 'keystore' config.  Must be valid URL.", e);
            }
        }

        return keyStoreURL;
    }
    
    protected static URL getFileAsURL(final String filename)
    {
        final File file = new File(filename);
        if (file.exists() && !file.isDirectory())
        {
            try
            {
                return file.toURL();
            }
            catch (final MalformedURLException murle) {} // ignore
        }
        return null ;
    }

    protected KeyStore getKeystore() throws ConfigurationException {
        URL url = getKeystoreURL();

        if(url != null) {
            try {
                InputStream keystoreStream = url.openStream();
                if(keystoreStream != null) {
                    return SSLUtil.loadKeyStore(keystoreStream, getKeystoreType(), getPasswordFromFile(getKeystorePassword()));
                } else {
                    throw new ConfigurationException("Unable to open keystore '" + url.toString() + "' for loading.");
                }
            } catch (IOException e) {
                throw new ConfigurationException("Unable to open keystore '" + url.toString() + "' for loading.", e);
            } catch (NoSuchAlgorithmException e) {
                throw new ConfigurationException("Unable to load keystore '" + url.toString() + "'.", e);
            } catch (KeyStoreException e) {
                throw new ConfigurationException("Unable to load keystore '" + url.toString() + "'.", e);
            } catch (CertificateException e) {
                throw new ConfigurationException("Unable to load keystore '" + url.toString() + "'.", e);
            }
        }

        return null;
    }

    protected KeyStore getTruststore() throws ConfigurationException {
        URL url = getTruststoreURL();

        if(url != null) {
            try {
                InputStream truststoreStream = url.openStream();
                if(truststoreStream != null) {
                    return SSLUtil.loadKeyStore(truststoreStream, getTruststoreType(), getPasswordFromFile(getTruststorePassword()));
                } else {
                    throw new ConfigurationException("Unable to open truststore '" + url.toString() + "' for loading.");
                }
            } catch (IOException e) {
                throw new ConfigurationException("Unable to open truststore '" + url.toString() + "' for loading.", e);
            } catch (NoSuchAlgorithmException e) {
                throw new ConfigurationException("Unable to load truststore '" + url.toString() + "'.", e);
            } catch (KeyStoreException e) {
                throw new ConfigurationException("Unable to load truststore '" + url.toString() + "'.", e);
            } catch (CertificateException e) {
                throw new ConfigurationException("Unable to load truststore '" + url.toString() + "'.", e);
            }
        }

        return null;
    }
    
    protected String getPasswordFromFile(final String password) throws ConfigurationException
    {
        if (PasswordUtil.isPasswordFile(password))
        {
            try
            {
                return new PasswordUtil(password).getPasswordAsString();
            }
            catch (final IOException e)
            {
                throw new ConfigurationException(e.getMessage(), e);
            }
        }
        return password;
    }
}
