/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.routing.http;

import java.io.IOException;

import org.apache.commons.httpclient.HttpMethodBase;
import org.apache.commons.httpclient.methods.ByteArrayRequestEntity;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.RequestEntity;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;

/**
 * HTTP POST Factory.  Handles messages whose payload is a byte[] by
 * converting the byte[] to a String; all other payloads are
 * stringified by calling their toString() method.
 * 
 * CONTENT_TYPE and CHARSET can be overridden in the ConfigTree.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class POSTHttpMethodFactory extends AbstractHttpMethodFactory {

	public static final String CONTENT_TYPE = "org.jboss.soa.esb.actions.routing.http.contentType";
	public static final String CHARSET = "org.jboss.soa.esb.actions.routing.http.charset";
	
    private MessagePayloadProxy payloadProxy;
    private String _contentType = null;
    private String _charset = null;

    @Override
    public void setConfiguration(ConfigTree config) throws ConfigurationException {
        super.setConfiguration(config) ;
        payloadProxy = new MessagePayloadProxy(config);
        
        _contentType = config.getAttribute(CONTENT_TYPE);
        _charset = config.getAttribute(CHARSET);
    }

    /**
     * Given the input message get the PostMethod.
     * 
     * Content-type and charset can be overridden at configuration time.
     */
    public HttpMethodBase getInstance(Message message) throws IOException {
        PostMethod method = new PostMethod( getEndpointPathAndQuery() );

        final RequestEntity entity;
        try
        {
            final Object payload = payloadProxy.getPayload(message);
            if (payload instanceof byte[])
                entity = new ByteArrayRequestEntity((byte[])payload, _contentType);
            else
                entity = new StringRequestEntity(String.valueOf(payload), _contentType, _charset);
        }
        catch (final MessageDeliverException e)
        {
             IOException ioe = new IOException("Failed to access message payload.");
             ioe.initCause(e);
             throw ioe;
        }

        method.setRequestEntity(entity);
        initialiseRetryHandler(method);
        return method;
    }

    /**
     * Given the serialized input message get the PostMethod.
     * 
     * @param messsage The serialized message.
     * @param contentType The content type for the message.
     * @param charset The charcter encoding for the message.
     * @return The PostMethod wrapping the serialized message.
     */
    public HttpMethodBase getMethod(String message, String contentType, String charset) throws IOException {
        PostMethod method = new PostMethod( getEndpointPathAndQuery() );

        final RequestEntity entity = new StringRequestEntity(message, contentType, charset);

        method.setRequestEntity(entity);
        initialiseRetryHandler(method);
        return method;
    }
}