/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.routing.email;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.routing.AbstractRouter;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;

/**
 * EmailRouter routes the ESB message payload to a configured email account.
 * <p/>
 * Like all router processing will terminate after this action has been processed.
 * 
 * Example configuration:
 * <pre>{@code
 * <action name="send-email" class="org.jboss.soa.esb.actions.routing.email.EmailRouter">
 *    <property name="host" value="localhost" />
 *    <property name="port" value="25" />
 *    <property name="username" value="danbev" />
 *    <property name="password" value="password" />
 *    <property name="from" value="jbossesb" />
 *    <property name="sendTo" value="danbev" />
 *    <property name="subject" value="quickstart routing" />
 *    <property name="unwrap" value="true" />
 * </action>
 * }</pre>
 * For details about the properties listed above please see {@link Emailer}s javadoc.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * @since 4.6
 *
 */
public class EmailRouter extends AbstractRouter
{
    private Emailer emailer;
    private ConfigTree config;

    public EmailRouter(ConfigTree config) throws ConfigurationException
    {
        super(config);
        emailer = new Emailer(config);
        this.config = config;
    }
    
    @Override
    public Message process(Message message) throws ActionProcessingException 
    {
        Emailer.overrideSmtpProperties(message, config);
        return super.process(message);
    }

    @Override
    public void route(Object object) throws ActionProcessingException
    {
        try
        {
            emailer.sendEmail(config, object);
        } 
        catch (MessageDeliverException e)
        {
            throw new ActionProcessingException("Exception while trying to send email.", e);
        }
    }

}
