/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.monitoring;

import java.util.ArrayList;
import java.util.Set;

import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;

import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.message.Message;

/**
 * Monitor store for any message that matches the MVEL pattern expression.     An ArrayList of 
 * MonitoringPatternBeans is stored, which can be reset.
 * 
 * @author tcunning
 * @since Version 4.4
 */
public class MVELMonitor implements MVELMonitorMBean {
	public ArrayList <MonitoringPatternBean> monitoredEvents; 
	private ConfigTree m_config;
	
	public MVELMonitor(ConfigTree config) {
		monitoredEvents = new ArrayList();
		m_config = config;
	}
	
	public ArrayList getMonitoredEvents() {
		return (ArrayList) monitoredEvents;
	}

	public void reset() {
		monitoredEvents = new ArrayList();
	}

	public void addEvent(Message message, MonitoringPatternBean mpb) {
		mpb.setMessageString(message.toString());
		monitoredEvents.add(mpb);
	}
	
	protected void registerMBean() throws InstanceAlreadyExistsException, 
		InstanceNotFoundException, MBeanRegistrationException, MBeanRegistrationException,
		MalformedObjectNameException, NotCompliantMBeanException {
        MBeanServer mbeanServer = null;
        try {
        	mbeanServer = MBeanServerLocator.locateJBoss();
        } catch (IllegalStateException ise) {
        	// If we can't find a JBoss MBeanServer, just return
        	// Needed for unit tests
        	return;
        }

        ObjectName listObjectName = getObjectName();

        if (mbeanServer.isRegistered(listObjectName)) {
        	mbeanServer.unregisterMBean(listObjectName);
        }
        
        mbeanServer.registerMBean(this, listObjectName);
	}
	
	protected ObjectName getObjectName() throws MalformedObjectNameException {
		ObjectName listObjectName = null;

		String deploymentName = m_config.getParent().getParent().getAttribute(ListenerTagNames.DEPLOYMENT_NAME_TAG);
		String serviceName = m_config.getParent().getAttribute(ListenerTagNames.SERVICE_NAME_TAG);

		StringBuffer objectName = new StringBuffer("category=MVELMonitor");
		if (deploymentName != null) {
			objectName.append(",").append(ListenerTagNames.DEPLOYMENT_NAME_TAG).append("=").append(deploymentName);
		}
			
		if (serviceName != null) {
			if (objectName.length() > 0) {
				objectName.append(",");
			}
			objectName.append(ListenerTagNames.SERVICE_NAME_TAG).append("=").append(serviceName);
		}
						
		listObjectName = new ObjectName("jboss.esb:" + objectName.toString());

		return listObjectName;
	}
}