/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.converters.xstream.conf;

import java.util.ArrayList;
import java.util.List;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;

import com.thoughtworks.xstream.XStream;

/**
 * XStream config is responsible for parsing a ConfigTree instance
 * and creating configuration objects. 
 * </p>
 * These configuration objects contain classes (if applicable) that will be loaded. 
 * Doing this at configuration time instead of runtime will save processing time.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class XStreamConfigurator
{
    private final ConfigTree config;
    
    private List<ClassAliasConf> classAliases;
    private List<FieldAliasConf> fieldAliases;
    private List<AttributeAliasConf> attributeAliases;
    private List<ImplicitCollectionConf> implicitCollections;
    private List<ConverterConf> converters;

    public XStreamConfigurator(final ConfigTree config)
    {
        AssertArgument.isNotNull(config, "config");
        this.config = config;
        classAliases = createClassAliases();
        fieldAliases = createFieldAliases();
        attributeAliases = createAttributeAliases();
        implicitCollections = createImplicitCollections();
        converters = createConverters();
    }
    
    public List<ClassAliasConf> getClassAliases()
    {
        return classAliases;
    }
    
    public List<FieldAliasConf> getFieldAliases()
    {
        return fieldAliases;
    }
    
    public List<AttributeAliasConf> getAttributeAliases()
    {
        return attributeAliases;
    }
    
    public List<ImplicitCollectionConf> getImplicitCollections()
    {
        return implicitCollections;
    }
    
    public List<ConverterConf> getConverters()
    {
        return converters;
    }
    
    private List<ClassAliasConf> createClassAliases()
    {
        final List<ClassAliasConf> aliases = new ArrayList<ClassAliasConf>();
        
        final ConfigTree[] children = config.getChildren("alias");
        if (children != null) 
        {
            for (final ConfigTree alias : children)
            {
                aliases.add(new ClassAliasConf(alias.getAttribute("name"), alias.getAttribute("class"), alias.getAttribute("defaultType")));
            }
        }
        return aliases;
    }
    
    private List<FieldAliasConf> createFieldAliases()
    {
        final List<FieldAliasConf> aliases = new ArrayList<FieldAliasConf>();
        
        final ConfigTree[] children = config.getChildren("field-alias");
        
        if (children != null) 
        {
            for (ConfigTree alias : children )
            {
                aliases.add(new FieldAliasConf(alias.getAttribute("alias"), alias.getAttribute("definedIn"), alias.getAttribute("fieldName")));
            }
        }
        return aliases;
    }
    
    private List<AttributeAliasConf> createAttributeAliases()
    {
        final List<AttributeAliasConf> aliases = new ArrayList<AttributeAliasConf>();
        
        final ConfigTree[] children = config.getChildren("attribute-alias");
        
        if (children != null) 
        {
            for (ConfigTree alias : children )
            {
                aliases.add(new AttributeAliasConf(alias.getAttribute("name"), alias.getAttribute("class")));
            }
        }
        return aliases;
    }
    
    private List<ImplicitCollectionConf> createImplicitCollections()
    {
        final List<ImplicitCollectionConf> imps = new ArrayList<ImplicitCollectionConf>();
        
        final ConfigTree[] children = config.getChildren("implicit-collection");
        if (children != null) 
        {
            for (ConfigTree alias : children)
            {
                imps.add(new ImplicitCollectionConf(alias.getAttribute("ownerType"), alias.getAttribute("fieldName"), alias.getAttribute("itemFieldName"), alias.getAttribute("itemType")));
            }
        }
        return imps;
    }
    
    private List<ConverterConf> createConverters()
    {
        List<ConverterConf> converters = new ArrayList<ConverterConf>();
        
        ConfigTree[] children = config.getChildren("converter");
        
        if ( children != null ) 
        {
            for (ConfigTree conv : children )
            {
                converters.add(new ConverterConf(conv.getAttribute("class"), conv.getAttribute("priority")));
            }
        }   
        return converters;
    }
    
    public static void addClassAliases(final List<ClassAliasConf> classAliases,final XStream xstream)
    {
        for (ClassAliasConf classAlias : classAliases)
        {
            if (classAlias.getDefaultType() == null )
            {
                xstream.alias(classAlias.getAlias(), classAlias.getType());
            }
            else
            {
                xstream.alias(classAlias.getAlias(), classAlias.getType(), classAlias.getDefaultType());
            }
        }
    }
    
    public static void addFieldAliases(final List<FieldAliasConf> fieldAliases, final XStream xstream) throws ActionProcessingException
    {
        if (fieldAliases == null )
            return;
        
        for (FieldAliasConf fieldAlias : fieldAliases)
        {
            xstream.aliasField(fieldAlias.getAlias(), fieldAlias.getDefinedIn(), fieldAlias.getFieldName());
        }
    }
    
    public static void addImplicitCollections(final List<ImplicitCollectionConf> implicitCollections, final XStream xstream) throws ActionProcessingException
    {
        if (implicitCollections == null )
            return;
        
        for (ImplicitCollectionConf impl : implicitCollections)
        {
            if (impl.getItemFieldName() != null && impl.getItemType() != null)
            {
                xstream.addImplicitCollection(impl.getOwnerType(), impl.getFieldName(), impl.getItemFieldName(), impl.getItemType());
            }
            else if ( impl.getItemType() != null)
            {
                xstream.addImplicitCollection(impl.getOwnerType(), impl.getFieldName(), impl.getItemType());
            }
            else
            {
                xstream.addImplicitCollection(impl.getOwnerType(), impl.getFieldName());
            }
        }
    }
    
    public static void addConverters(final List<ConverterConf> converters, final XStream xstream) throws ActionProcessingException
    {
        if (converters == null )
            return;
        
        for (ConverterConf conf : converters)
        {
            if (conf.getPriority() == -1)
            {
                xstream.registerConverter(conf.getConverter());
            }
            else
            {
                xstream.registerConverter(conf.getConverter(), conf.getPriority());
            }
        }
    }
}
