/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.cbr;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.jboss.soa.esb.util.XPathNamespaceContext;

import com.envoisolutions.sxc.xpath.XPathBuilder;
import com.envoisolutions.sxc.xpath.XPathEvaluator;
import com.envoisolutions.sxc.xpath.XPathEvent;
import com.envoisolutions.sxc.xpath.XPathEventHandler;
import com.envoisolutions.sxc.xpath.XPathException;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.util.*;

import javax.xml.stream.XMLStreamException;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;

/**
 * <a href="http://sxc.codehaus.org/XPath">SXC XPath</a> Content Based Router implementation.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SxcXPathRouter extends AbstractPropertyRulesRouter {

	private XPathNamespaceContext namespaceContext;

    public void setConfigTree(ConfigTree configTree) throws MessageRouterException {
        try {
            namespaceContext = new NamespaceContext(configTree.getChildren("namespace"));
        } catch (ConfigurationException e) {
            throw new MessageRouterException("Error loading namespace prefix mappings.", e);
        }

        super.setConfigTree(configTree);
    }

    public Map<String, RoutingRule> buildRoutingMap(Properties rules) throws MessageRouterException {
        Map<String, RoutingRule> routingMap = new HashMap<String, RoutingRule>();

        Set<Map.Entry<Object, Object>> ruleSet = rules.entrySet();
        for(Map.Entry<Object, Object> rule : ruleSet) {
            String destinationName = (String) rule.getKey();
            String expression = (String) rule.getValue();

            try {
            	routingMap.put(destinationName, new SxcXpathRoutingRule(expression, namespaceContext));
            } catch(XPathException e) {
            	throw new MessageRouterException("Error compiling XPath expression '" + expression + "'.", e);
            }
        }

        return routingMap;
    }

    public javax.xml.namespace.NamespaceContext getNamespaceContext() {
        return namespaceContext;
    }

    private static class SxcXpathRoutingRule implements RoutingRule {

        private XPathEvaluator evaluator;
		private String expression;
        private static ThreadLocal<Boolean> matchTL = new ThreadLocal<Boolean>();

		private SxcXpathRoutingRule(String expression, XPathNamespaceContext namespaceContext) {
			this.expression = expression;
			
        	XPathEventHandler matchIdHandler = new XPathEventHandler() {
        	    public void onMatch(XPathEvent event) throws XMLStreamException {
        	    	matchTL.set(true);
        	    }
        	};
        	
        	XPathBuilder builder = new XPathBuilder();
    		builder.addAllPrefixes(namespaceContext.getPrefixToURI());
        	builder.listen(expression, matchIdHandler);

        	evaluator = builder.compile();
        }

        public boolean evaluate(Object objectToTest) throws MessageRouterException {

        	try {
        		// Need to use a ThreadLocal because the SXC XPathEvaluator class
        		// doesn't support a context of any kind... so using the executing
        		// thread as our context...
        		matchTL.set(false);
        		evaluator.evaluate(toSource(objectToTest));
				return matchTL.get();
			} catch (Exception e) {
				throw new MessageRouterException("Exception while evaluating SXC expression '" + expression + "'");
			}
        }

		private Source toSource(Object objectToTest) throws MessageRouterException {
            if(objectToTest instanceof String) {
                return new StreamSource(new StringReader((String) objectToTest));
            } else if(objectToTest instanceof byte[]) {
                return new StreamSource(new ByteArrayInputStream((byte[]) objectToTest));
            } else if(objectToTest instanceof Reader) {
                return new StreamSource((Reader) objectToTest);
            } else if(objectToTest instanceof InputStream) {
                return new StreamSource((InputStream) objectToTest);
//            } else if(objectToTest instanceof Node) {
            	// Disabled.... with dom (at least) SXC has validation turned on, so if the message does not define an XSD/DTD, you get errors
            	// Can't see how to turn off validation (probably a STAX thing)... API not exposed (that I can see anyway).
                
//                return new DOMSource((Node)objectToTest);
            } else {
                throw new MessageRouterException("Unsupported SXC CBR payload type '" + objectToTest.getClass().getName() + "'.");
            }
		}
    }
}