/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

/**
 * Routes the Message argument to a fixed list of services ([category,name]) 
 * @author <a href="mailto:schifest@heuristica.com.ar">schifest@heuristica.com.ar</a>
 * @author Kevin.Conner@jboss.com
 * @since Version 4.0
 */
package org.jboss.soa.esb.actions;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.client.MessageMulticaster;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.registry.RegistryException;

public class StaticWiretap extends AbstractActionPipelineProcessor
{
    public static final String ROUTE_TO_TAG = "route-to";

    public StaticWiretap(ConfigTree config) throws ConfigurationException,
        RegistryException
    {
        _config = config;
    }

    /**
     * Routes the message to one or more destinations, returning original message.
     */
    public Message process(Message message) throws ActionProcessingException
    {
        try
        {
            messageMulticaster.sendToAll(message);
            return message;
        } catch (RegistryException e) {
            throw new ActionProcessingException(e);
        } catch (MessageDeliverException e) {
            throw new ActionProcessingException(e);
        }
    }

    /**
     * Initialization by reading the configuration.
     */
    public void initialise() throws ActionLifecycleException
    {
        messageMulticaster = new MessageMulticaster(_config.getAttribute("action", "%unset%"));
        messageMulticaster.setAggregatorOnProperties(Aggregator.aggregatorOnProperties(_config));

        ConfigTree[] destList = _config.getChildren(ROUTE_TO_TAG);
        if (null == destList || destList.length < 1)
        {
            _logger.warn("Missing or empty destination list - This action class won't have any effect");
            return;
        }
        for (ConfigTree curr : destList)
        {
            try
            {
                String category = curr.getAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG, "");
                String name = curr.getRequiredAttribute(ListenerTagNames.SERVICE_NAME_TAG);
                Service service = new Service(category, name);
                messageMulticaster.addRecipient(service);
            }
            catch (Exception e)
            {
                throw new ActionLifecycleException("Problems with destination list", e);
            }
        }
    }

    protected ConfigTree _config;

    protected MessageMulticaster messageMulticaster;

    protected static final Logger _logger = Logger.getLogger(StaticWiretap.class);
}
