/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.internal.soa.esb.util.stax.events;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.XMLConstants;
import javax.xml.namespace.NamespaceContext;
import javax.xml.stream.events.Namespace;

/**
 * NamespaceContext for events.
 */
public class ESBStaxNamespaceContext implements NamespaceContext
{
    private final Map<String, String> namespaceToPrefix ;
    private final Map<String, String> prefixToNamespace ;

    public ESBStaxNamespaceContext(final List<Namespace> namespaces)
    {
        final int size = (namespaces == null ? 0 : namespaces.size()) ;
        if (size == 0)
        {
            namespaceToPrefix = Collections.emptyMap() ;
            prefixToNamespace = Collections.emptyMap() ;
        }
        else
        {
            namespaceToPrefix = new HashMap<String, String>(size) ;
            prefixToNamespace = new HashMap<String, String>(size) ;
            for(Namespace namespace: namespaces)
            {
                final String prefix = namespace.getPrefix() ;
                final String namespaceURI = namespace.getNamespaceURI() ;
                namespaceToPrefix.put(namespaceURI, prefix) ;
                prefixToNamespace.put(prefix, namespaceURI) ;
            }
        }
    }

    static NamespaceContext getNamespaceContext(final List<Namespace> namespaces)
    {
        if (namespaces == null)
        {
            return EMPTY_NAMESPACE_CONTEXT ;
        }
        else if (namespaces.size() > 0)
        {
            return new ESBStaxNamespaceContext(namespaces) ;
        }
        else
        {
            return EMPTY_NAMESPACE_CONTEXT ;
        }
    }

    public String getNamespaceURI(final String prefix)
    {
        return normalize(prefixToNamespace.get(prefix), XMLConstants.NULL_NS_URI);
    }

    public String getPrefix(final String namespaceURI)
    {
        return normalize(namespaceToPrefix.get(namespaceURI), XMLConstants.DEFAULT_NS_PREFIX);
    }

    public Iterator<String> getPrefixes(final String namespaceURI)
    {
        final String prefix = namespaceToPrefix.get(namespaceURI) ;
        if (prefix == null)
        {
            final List<String> emptyList = Collections.emptyList() ;
            return emptyList.iterator() ;
        }
        else
        {
            return Arrays.asList(prefix).iterator() ;
        }
    }

    private static String normalize(final String value, final String defaultValue)
    {
        return (value == null ? defaultValue : value) ;
    }

    private static final List<Namespace> EMPTY_NAMESPACES = Collections.emptyList() ;
    private static final NamespaceContext EMPTY_NAMESPACE_CONTEXT = new ESBStaxNamespaceContext(EMPTY_NAMESPACES) ;
}
