/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.services.registry;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.InVMEpr;
import org.jboss.soa.esb.services.registry.AbstractRegistryInterceptor;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;
import org.jboss.internal.soa.esb.couriers.transport.InVMException;
import org.jboss.internal.soa.esb.couriers.transport.InVMTransport;

import java.util.List;

/**
 * In VM registry.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class InVMRegistryInterceptor extends AbstractRegistryInterceptor {

    public void registerEPR(String serviceCategoryName, String serviceName, String serviceDescription, EPR epr, String eprDescription) throws RegistryException {
        if (epr instanceof InVMEpr) {
            try {
                InVMTransport.getInstance().registerEPR(serviceCategoryName, serviceName, (InVMEpr)epr) ;
            } catch (final InVMException invme) {
                throw new RegistryException("Unexpected error registering service EPR", invme) ;
            }
        } else {
            getRegistry().registerEPR(serviceCategoryName, serviceName, serviceDescription, epr, eprDescription) ;
        }
    }

    public void unRegisterService(String category, String serviceName) throws RegistryException, ServiceNotFoundException {
        final boolean entryRemoved = InVMTransport.getInstance().unRegisterService(category, serviceName) ;
        try {
            getRegistry().unRegisterService(category, serviceName) ;
        } catch (final ServiceNotFoundException snfe) {
            if (!entryRemoved) {
                throw snfe ;
            }
        }
    }

    public void unRegisterEPR(String serviceCategoryName, String serviceName, EPR epr) throws RegistryException, ServiceNotFoundException {
        if (epr instanceof InVMEpr) {
            InVMTransport.getInstance().unRegisterEPR(serviceCategoryName, serviceName, (InVMEpr)epr);
        } else {
            getRegistry().unRegisterEPR(serviceCategoryName, serviceName, epr) ;
        }
    }

    public List<String> findAllServices() throws RegistryException {
        final List<String> services = InVMTransport.getInstance().findAllServices();
        services.addAll(getRegistry().findAllServices()) ;
        return services ;
    }

    public List<String> findServices(String serviceCategoryName) throws RegistryException {
        final List<String> services = InVMTransport.getInstance().findServices(serviceCategoryName);
        services.addAll(getRegistry().findServices(serviceCategoryName)) ;
        return services ;
    }
    
    public List<EPR> findEPRs(String serviceCategoryName, String serviceName) throws RegistryException, ServiceNotFoundException {
        List<EPR> eprs = InVMTransport.getInstance().findEPRs(serviceCategoryName, serviceName) ;
        if (eprs.size() == 0) {
            return getRegistry().findEPRs(serviceCategoryName, serviceName) ;
        } else {
            try {
                eprs.addAll(getRegistry().findEPRs(serviceCategoryName, serviceName)) ;
            } catch (final ServiceNotFoundException snfe) {
                // do nothing, we already have some EPRs.
            }
        }
        return eprs;
    }

    public EPR findEPR(String serviceCategoryName, String serviceName) throws RegistryException, ServiceNotFoundException {
        final EPR epr = InVMTransport.getInstance().findEPR(serviceCategoryName, serviceName);

        if(epr != null) {
            return epr;
        } else {
            return getRegistry().findEPR(serviceCategoryName, serviceName);
        }
    }
}
