/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.internal.soa.esb.rosetta.pooling;

import java.util.concurrent.atomic.AtomicBoolean;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;

/**
 * Wrapped message consumer class.
 */
class WrappedMessageConsumer implements MessageConsumer
{
    /**
     * The connection pool associated with the session.
     */
    protected final JmsConnectionPool connectionPool ;
    /**
     * The session associated with the consumer.
     */
    protected final JmsSession jmsSession ;
    /**
     * The message consumer.
     */
    private final MessageConsumer messageConsumer ;
    /**
     * valid flag for xa associations.
     */
    private final AtomicBoolean wrapperValid ;

    /**
     * Construct the wrapped message consumer.
     * @param connectionPool The connection pool associated with the session.
     * @param jmsSession The session associated with the consumer.
     * @param messageConsumer The message consumer.
     */
    public WrappedMessageConsumer(final JmsConnectionPool connectionPool,
        final JmsSession jmsSession, final MessageConsumer messageConsumer)
    {
        this(connectionPool, jmsSession, messageConsumer, null) ;
    }

    /**
     * Construct the wrapped message consumer.
     * @param connectionPool The connection pool associated with the session.
     * @param jmsSession The session associated with the consumer.
     * @param messageConsumer The message consumer.
     * @param wrapperValid valid flag for xa associations.
     */
    public WrappedMessageConsumer(final JmsConnectionPool connectionPool,
        final JmsSession jmsSession, final MessageConsumer messageConsumer,
        final AtomicBoolean wrapperValid)
    {
        this.connectionPool = connectionPool ;
        this.jmsSession = jmsSession ;
        this.messageConsumer = messageConsumer ;
        this.wrapperValid = wrapperValid ;
    }

    public void close()
        throws JMSException
    {
        validate() ;
        try
        {
            messageConsumer.close();
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public MessageListener getMessageListener()
        throws JMSException
    {
        associate() ;
        try
        {
            return messageConsumer.getMessageListener() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public String getMessageSelector()
        throws JMSException
    {
        associate() ;
        try
        {
            return messageConsumer.getMessageSelector() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public Message receive()
        throws JMSException
    {
        associate() ;
        try
        {
            return messageConsumer.receive() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public Message receive(final long arg0)
        throws JMSException
    {
        associate() ;
        try
        {
            return messageConsumer.receive(arg0) ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public Message receiveNoWait()
        throws JMSException
    {
        associate() ;
        try
        {
            return messageConsumer.receiveNoWait() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public void setMessageListener(final MessageListener listener)
        throws JMSException
    {
        associate() ;
        try
        {
            messageConsumer.setMessageListener(listener) ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    private void validate()
        throws JMSException
    {
        if ((wrapperValid != null) && !wrapperValid.get())
        {
            throw new JMSException("Instance is no longer valid") ;
        }
    }

    protected void associate()
        throws JMSException
    {
        validate() ;
        jmsSession.associate() ;
    }
}
