/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.internal.soa.esb.registry.client;

import java.net.URL;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import javax.xml.namespace.QName;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.Service;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.juddi.v3.client.config.UDDIClerkManager;
import org.apache.juddi.v3.client.config.UDDIClientContainer;
import org.apache.juddi.v3.client.config.UDDINode;
import org.apache.juddi.v3.client.transport.Transport;
import org.apache.juddi.v3.client.transport.TransportException;
import org.apache.juddi.v3_service.JUDDIApiPortType;
import org.jboss.soa.esb.util.ClassUtil;
import org.uddi.v3_service.UDDICustodyTransferPortType;
import org.uddi.v3_service.UDDIInquiryPortType;
import org.uddi.v3_service.UDDIPublicationPortType;
import org.uddi.v3_service.UDDISecurityPortType;
import org.uddi.v3_service.UDDISubscriptionListenerPortType;
import org.uddi.v3_service.UDDISubscriptionPortType;

public class JuddiJAXWSTransport extends Transport
{
    /**
     * The base resource path for the WSDL resources.
     */
    private static final String BASE_RESOURCE_PATH = "wsdl/" ;
    /**
     * The UDDI namespace.
     */
    public final static String NAMESPACE_UDDI_V3 = "urn:uddi-org:v3_service";
    /**
     * The jUDDI v3 namespace.
     */
    public final static String NAMESPACE_JUDDI_V3 = "urn:juddi-apache-org:v3_service";
    
    /**
     * The jUDDI v3 service wsdl for inquiry.
     */
    public final static URL WSDL_INQUIRY_SERVICE = getResource("esb_inquiry_service.wsdl");
    /**
     * The jUDDI v3 service name for inquiry.
     */
    public final static String NAME_INQUIRY_SERVICE = "UDDI_Inquiry_Port";
    /**
     * The jUDDI v3 service qname for inquiry.
     */
    public final static QName QNAME_INQUIRY_SERVICE = new QName(NAMESPACE_UDDI_V3, NAME_INQUIRY_SERVICE) ;
    /**
     * The jUDDI v3 service wsdl for security.
     */
    public final static URL WSDL_SECURITY_SERVICE = getResource("esb_security_service.wsdl") ;
    /**
     * The jUDDI v3 service name for security.
     */
    public final static String NAME_SECURITY_SERVICE = "UDDI_Security_Port";
    /**
     * The jUDDI v3 service qname for security.
     */
    public final static QName QNAME_SECURITY_SERVICE = new QName(NAMESPACE_UDDI_V3, NAME_SECURITY_SERVICE) ;
    /**
     * The jUDDI v3 service wsdl for publish.
     */
    public final static URL WSDL_PUBLISH_SERVICE = getResource("esb_publication_service.wsdl") ;
    /**
     * The jUDDI v3 service name for publish.
     */
    public final static String NAME_PUBLISH_SERVICE = "UDDI_Publish_Port";
    /**
     * The jUDDI v3 service qname for publish.
     */
    public final static QName QNAME_PUBLISH_SERVICE = new QName(NAMESPACE_UDDI_V3, NAME_PUBLISH_SERVICE) ;
    /**
     * The jUDDI v3 service wsdl for subscription.
     */
    public final static URL WSDL_SUBSCRIPTION_SERVICE = getResource("esb_subscription_service.wsdl") ;
    /**
     * The jUDDI v3 service name for subscription.
     */
    public final static String NAME_SUBSCRIPTION_SERVICE = "UDDI_Subscription_Port";
    /**
     * The jUDDI v3 service qname for subscription.
     */
    public final static QName QNAME_SUBSCRIPTION_SERVICE = new QName(NAMESPACE_UDDI_V3, NAME_SUBSCRIPTION_SERVICE) ;
    /**
     * The jUDDI v3 service wsdl for subscription listener.
     */
    public final static URL WSDL_SUBSCRIPTION_LISTENER_SERVICE = getResource("esb_subscriptionlistener_service.wsdl") ;
    /**
     * The jUDDI v3 service name for subscription listener.
     */
    public final static String NAME_SUBSCRIPTION_LISTENER_SERVICE = "UDDI_SubscriptionListener_Port";
    /**
     * The jUDDI v3 service qname for subscription listener.
     */
    public final static QName QNAME_SUBSCRIPTION_LISTENER_SERVICE = new QName(NAMESPACE_UDDI_V3, NAME_SUBSCRIPTION_LISTENER_SERVICE) ;
    /**
     * The jUDDI v3 service wsdl for custody transfer.
     */
    public final static URL WSDL_CUSTODY_TRANSFER_SERVICE = getResource("esb_custodytransfer_service.wsdl") ;
    /**
     * The jUDDI v3 service name for custody transfer.
     */
    public final static String NAME_CUSTODY_TRANSFER_SERVICE = "UDDI_CustodyTransfer_Port";
    /**
     * The jUDDI v3 service qname for custody transfer.
     */
    public final static QName QNAME_CUSTODY_TRANSFER_SERVICE = new QName(NAMESPACE_UDDI_V3, NAME_CUSTODY_TRANSFER_SERVICE) ;
    /**
     * The jUDDI v3 service wsdl for jUDDI API.
     */
    public final static URL WSDL_JUDDI_API_SERVICE = getResource("esb_juddi_api_service.wsdl") ;
    /**
     * The jUDDI v3 service name for jUDDI API.
     */
    public final static String NAME_JUDDI_API_SERVICE = "JUDDI_Api_Port";
    /**
     * The jUDDI v3 service qname for jUDDI API.
     */
    public final static QName QNAME_JUDDI_API_SERVICE = new QName(NAMESPACE_JUDDI_V3, NAME_JUDDI_API_SERVICE) ;
    
    /**
     * The manager name to use for this transport.
     */
    private final String managerName ;
    /**
     * The node name to use for this transport.
     */
    private final String nodeName ;
    
    /**
     * The api service instance.
     */
    private final AtomicReference<JUDDIApiPortType> apiService = new AtomicReference<JUDDIApiPortType>() ;
    /**
     * The custody transfer service instance.
     */
    private final AtomicReference<UDDICustodyTransferPortType> custodyTransferService = new AtomicReference<UDDICustodyTransferPortType>() ;
    /**
     * The inquiry service instance.
     */
    private final AtomicReference<UDDIInquiryPortType> inquiryService = new AtomicReference<UDDIInquiryPortType>() ;
    /**
     * The publish service instance.
     */
    private final AtomicReference<UDDIPublicationPortType> publishService = new AtomicReference<UDDIPublicationPortType>() ;
    /**
     * The security service instance.
     */
    private final AtomicReference<UDDISecurityPortType> securityService = new AtomicReference<UDDISecurityPortType>() ;
    /**
     * The subscription listener service.
     */
    private final AtomicReference<UDDISubscriptionListenerPortType> subscriptionListenerService = new AtomicReference<UDDISubscriptionListenerPortType>() ;
    /**
     * The subscription service.
     */
    private final AtomicReference<UDDISubscriptionPortType> subscriptionService = new AtomicReference<UDDISubscriptionPortType>() ;

    /**
     * Default constructor.
     */
    public JuddiJAXWSTransport()
        throws ConfigurationException
    {
        this(null) ;
    }

    /**
     * Construct a transport for the specified node.
     * @param nodeName The node name.
     */
    public JuddiJAXWSTransport(final String nodeName)
        throws ConfigurationException
    {
        this(null, nodeName) ;
    }

    /**
     * Construct a transport for the specified node.
     * @param managerName The manager name.
     * @param nodeName The node name.
     */
    public JuddiJAXWSTransport(final String managerName, final String nodeName)
        throws ConfigurationException
    {
        this.managerName = managerName ;
        this.nodeName = (nodeName == null ? Transport.DEFAULT_NODE_NAME : nodeName) ;
    }
    
    @Override
    public JUDDIApiPortType getJUDDIApiService(final String endpointURL)
        throws TransportException
    {
        final JUDDIApiPortType currentApiService = apiService.get() ;
        if (currentApiService != null)
        {
            return currentApiService ;
        }
        final String apiURL = getUDDINode(managerName, nodeName).getJuddiApiUrl() ;
        
        final Service service = Service.create(WSDL_JUDDI_API_SERVICE, QNAME_JUDDI_API_SERVICE) ;
        final JUDDIApiPortType newApiService = (JUDDIApiPortType) service.getPort(JUDDIApiPortType.class) ;
        initialiseEndpoint(newApiService, apiURL) ;
        if (apiService.compareAndSet(null, newApiService))
        {
            return newApiService ;
        }
        else
        {
            return apiService.get() ;
        }
    }

    @Override
    public UDDICustodyTransferPortType getUDDICustodyTransferService(final String endpointURL)
        throws TransportException
    {
        final UDDICustodyTransferPortType currentCustodyTransferService = custodyTransferService.get() ;
        if (currentCustodyTransferService != null)
        {
            return currentCustodyTransferService ;
        }
        final String custodyTransferURL = getUDDINode(managerName, nodeName).getCustodyTransferUrl() ;
        final Service service = Service.create(WSDL_CUSTODY_TRANSFER_SERVICE, QNAME_CUSTODY_TRANSFER_SERVICE) ;
        final UDDICustodyTransferPortType newCustodyTransferService = (UDDICustodyTransferPortType) service.getPort(UDDICustodyTransferPortType.class) ;
        initialiseEndpoint(newCustodyTransferService, custodyTransferURL) ;
        if (custodyTransferService.compareAndSet(null, newCustodyTransferService))
        {
            return newCustodyTransferService ;
        }
        else
        {
            return custodyTransferService.get() ;
        }
    }

    @Override
    public UDDIInquiryPortType getUDDIInquiryService(final String endpointURL)
        throws TransportException
    {
        final UDDIInquiryPortType currentInquiryService = inquiryService.get() ;
        if (currentInquiryService != null)
        {
            return currentInquiryService ;
        }
        final String inquiryURL = getUDDINode(managerName, nodeName).getInquiryUrl() ;
        final Service service = Service.create(WSDL_INQUIRY_SERVICE, QNAME_INQUIRY_SERVICE) ;
        final UDDIInquiryPortType newInquiryService = (UDDIInquiryPortType) service.getPort(UDDIInquiryPortType.class) ;
        initialiseEndpoint(newInquiryService, inquiryURL) ;
        if (inquiryService.compareAndSet(null, newInquiryService))
        {
            return newInquiryService ;
        }
        else
        {
            return inquiryService.get() ;
        }
    }

    @Override
    public UDDIPublicationPortType getUDDIPublishService(final String endpointURL)
            throws TransportException
    {
        final UDDIPublicationPortType currentPublishService = publishService.get() ;
        if (currentPublishService != null)
        {
            return currentPublishService ;
        }
        final String publishURL = getUDDINode(managerName, nodeName).getPublishUrl() ;
        final Service service = Service.create(WSDL_PUBLISH_SERVICE, QNAME_PUBLISH_SERVICE) ;
        final UDDIPublicationPortType newPublishService = (UDDIPublicationPortType) service.getPort(UDDIPublicationPortType.class) ;
        initialiseEndpoint(newPublishService, publishURL) ;
        if (publishService.compareAndSet(null, newPublishService))
        {
            return newPublishService ;
        }
        else
        {
            return publishService.get() ;
        }
    }

    @Override
    public UDDISecurityPortType getUDDISecurityService(final String endpointURL)
            throws TransportException
    {
        final UDDISecurityPortType currentSecurityService = securityService.get() ;
        if (currentSecurityService != null)
        {
            return currentSecurityService ;
        }
        final String securityURL = getUDDINode(managerName, nodeName).getSecurityUrl() ;
        final Service service = Service.create(WSDL_SECURITY_SERVICE, QNAME_SECURITY_SERVICE) ;
        final UDDISecurityPortType newSecurityService = (UDDISecurityPortType) service.getPort(UDDISecurityPortType.class) ;
        initialiseEndpoint(newSecurityService, securityURL) ;
        if (securityService.compareAndSet(null, newSecurityService))
        {
            return newSecurityService ;
        }
        else
        {
            return securityService.get() ;
        }
    }

    @Override
    public UDDISubscriptionListenerPortType getUDDISubscriptionListenerService(final String endpointURL)
        throws TransportException
    {
        final UDDISubscriptionListenerPortType currentSubscriptionListenerService = subscriptionListenerService.get() ;
        if (currentSubscriptionListenerService != null)
        {
            return currentSubscriptionListenerService ;
        }
        final String subscriptionListenerURL = getUDDINode(managerName, nodeName).getSubscriptionListenerUrl() ;
        final Service service = Service.create(WSDL_SUBSCRIPTION_LISTENER_SERVICE, QNAME_SUBSCRIPTION_LISTENER_SERVICE) ;
        final UDDISubscriptionListenerPortType newSubscriptionListenerService = (UDDISubscriptionListenerPortType) service.getPort(UDDISubscriptionListenerPortType.class) ;
        initialiseEndpoint(newSubscriptionListenerService, subscriptionListenerURL) ;
        if (subscriptionListenerService.compareAndSet(null, newSubscriptionListenerService))
        {
            return newSubscriptionListenerService ;
        }
        else
        {
            return subscriptionListenerService.get() ;
        }
    }

    @Override
    public UDDISubscriptionPortType getUDDISubscriptionService(final String endpointURL)
        throws TransportException
    {
        final UDDISubscriptionPortType currentSubscriptionService = subscriptionService.get() ;
        if (currentSubscriptionService != null)
        {
            return currentSubscriptionService ;
        }
        final String subscriptionURL = getUDDINode(managerName, nodeName).getSubscriptionUrl() ;
        final Service service = Service.create(WSDL_SUBSCRIPTION_SERVICE, QNAME_SUBSCRIPTION_SERVICE) ;
        final UDDISubscriptionPortType newSubscriptionService = (UDDISubscriptionPortType) service.getPort(UDDISubscriptionPortType.class) ;
        initialiseEndpoint(newSubscriptionService, subscriptionURL) ;
        if (subscriptionService.compareAndSet(null, newSubscriptionService))
        {
            return newSubscriptionService ;
        }
        else
        {
            return subscriptionService.get() ;
        }
    }

    private static UDDINode getUDDINode(final String managerName, final String nodeName)
        throws TransportException
    {
        try
        {
            final UDDIClerkManager manager = UDDIClientContainer.getUDDIClerkManager(managerName) ;
            return manager.getClientConfig().getUDDINode(nodeName) ;
        }
        catch (final ConfigurationException ce)
        {
            throw new TransportException("Failed to retrieve UDDI node: " + nodeName, ce) ;
        }
    }

    private static void initialiseEndpoint(final Object endpoint, final String endpointURL)
        throws TransportException
    {
        if (!(endpoint instanceof BindingProvider))
        {
            throw new TransportException("Endpoint does not implement BindingProvider interface") ;
        }
        final BindingProvider provider = (BindingProvider)endpoint ;
        final Map<String, Object> requestContext = provider.getRequestContext() ;
        requestContext.put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, endpointURL) ;
    }

    private static URL getResource(final String resource)
    {
        return ClassUtil.getResource(BASE_RESOURCE_PATH + resource, JuddiJAXWSTransport.class) ;
    }
}
