/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.registry.client;

import org.apache.juddi.v3.client.transport.Transport;
import org.apache.juddi.v3.client.transport.TransportException;
import org.apache.juddi.v3_service.JUDDIApiPortType;
import org.uddi.v3_service.UDDICustodyTransferPortType;
import org.uddi.v3_service.UDDIInquiryPortType;
import org.uddi.v3_service.UDDIPublicationPortType;
import org.uddi.v3_service.UDDISecurityPortType;
import org.uddi.v3_service.UDDISubscriptionListenerPortType;
import org.uddi.v3_service.UDDISubscriptionPortType;

/**
 * InVM transport specific to the ESB installation.
 * 
 * There is no interface for this service so we extend the jUDDI transport class
 * and delegate to another implementation.
 */
public class JuddiInVMTransport extends Transport
{
    /**
     * The implementation of the ESB specific InVM Transport.
     */
    private static ESBInVMTransport implementation ;
    
    /**
     * The manager name of the current transport.
     */
    private final String managerName ;
    /**
     * The node name of the current transport.
     */
    private final String nodeName ;
    
    /**
     * Default constructor.
     */
    public JuddiInVMTransport()
    {
        this(null) ;
    }
    
    /**
     * Construct a transport for the specified node.
     * @param nodeName The node name.
     */
    public JuddiInVMTransport(final String nodeName)
    {
        this(null, nodeName) ;
    }
    
    /**
     * Construct a transport for the specified node.
     * @param managerName The manager name.
     * @param nodeName The node name.
     */
    public JuddiInVMTransport(final String managerName, final String nodeName)
    {
        this.managerName = managerName ;
        this.nodeName = (nodeName != null ? nodeName : Transport.DEFAULT_NODE_NAME) ;
    }
    
    @Override
    public JUDDIApiPortType getJUDDIApiService(final String endpointURL)
        throws TransportException
    {
        return getImplementation().getJUDDIApiService(managerName, nodeName, endpointURL) ;
    }

    @Override
    public UDDICustodyTransferPortType getUDDICustodyTransferService(final String endpointURL)
        throws TransportException
    {
        return getImplementation().getUDDICustodyTransferService(managerName, nodeName, endpointURL) ;
    }

    @Override
    public UDDIInquiryPortType getUDDIInquiryService(final String endpointURL)
        throws TransportException
    {
        return getImplementation().getUDDIInquiryService(managerName, nodeName, endpointURL) ;
    }

    @Override
    public UDDIPublicationPortType getUDDIPublishService(final String endpointURL)
            throws TransportException
    {
        return getImplementation().getUDDIPublishService(managerName, nodeName, endpointURL) ;
    }

    @Override
    public UDDISecurityPortType getUDDISecurityService(final String endpointURL)
            throws TransportException
    {
        return getImplementation().getUDDISecurityService(managerName, nodeName, endpointURL) ;
    }

    @Override
    public UDDISubscriptionListenerPortType getUDDISubscriptionListenerService(final String endpointURL)
        throws TransportException
    {
        return getImplementation().getUDDISubscriptionListenerService(managerName, nodeName, endpointURL) ;
    }

    @Override
    public UDDISubscriptionPortType getUDDISubscriptionService(final String endpointURL)
        throws TransportException
    {
        return getImplementation().getUDDISubscriptionService(managerName, nodeName, endpointURL) ;
    }
    
    /**
     * Get the current implementation.
     * @return The current implementation.
     * @throws TransportException if no implementation has been configured.
     */
    public static synchronized ESBInVMTransport getImplementation()
        throws TransportException
    {
        if (implementation == null)
        {
            throw new TransportException("Transport implementation not yet initialised") ;
        }
        return implementation ;
    }
    
    /**
     * Set the current implementation.
     * @param implementation The current implementation.
     */
    public static synchronized void setImplementation(final ESBInVMTransport implementation)
    {
        JuddiInVMTransport.implementation = implementation ;
    }
}
