package org.jboss.internal.soa.esb.parameters;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import javax.naming.CompoundName;
import javax.naming.InvalidNameException;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.parameters.ParamRepository;
import org.jboss.soa.esb.parameters.ParamRepositoryException;

/**
 * This class provides basic file-based storage/retrieval of parameter trees for
 * the JBoss ESB. <p/> This class creates a hierarchical parameter file
 * structure on the file system. E.g. the value for a parameter named
 * "org/jboss/param1" is stored in a file called "param1" in the folder "<root>/org/jboss",
 * where "root" is either the working directory (default) or the directory
 * specified in the "org.jboss.soa.esb.paramsRepository.file.root" System
 * property.
 * 
 * @author Esteban
 * 
 */
public class ParamFileRepository implements ParamRepository
{

	/**
	 * System property defining the repository root directory.
	 */
	public static final String FILE_PARAMS_REPOS_ROOT = "org.jboss.soa.esb.paramsRepository.file.root";

	/**
	 * The repository root dir.
	 */
	private File root;

	/**
	 * Logger.
	 */
	private Logger logger = Logger.getLogger(ParamFileRepository.class);

	/**
	 * Public default constructor.
	 */
	public ParamFileRepository()
	{
		String rootDir = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE)
				.getProperty(FILE_PARAMS_REPOS_ROOT);

		// Set the repository root directory.
		if (rootDir == null)
		{
			// set it to the working dir.
			root = new File("./");
		} else
		{
			root = new File(rootDir);
			if (!root.exists())
			{
				throw new IllegalStateException(
						"Nonexistant directory specified in the ["
								+ FILE_PARAMS_REPOS_ROOT
								+ "] System property: "
								+ root.getAbsolutePath());
			}
		}

		logger.info("Setting parameter repository root dir to ["
				+ root.getAbsolutePath() + "].");
	}

	public void add(String name, String value) throws ParamRepositoryException
	{
		FileParamName nameParam = asserNameOK(name);
		if (value == null)
		{
			throw new IllegalArgumentException(
					"null 'value' arg in method call.");
		}

		synchronized (this)
		{
			File paramFile = toParamFile(nameParam);
			File parentDir = paramFile.getParentFile();
			if (null == parentDir)
				parentDir = new File("");

			boolean result = parentDir.mkdirs();
			FileOutputStream fileStream = null;
			try
			{
				fileStream = new FileOutputStream(paramFile);
				fileStream.write(value.getBytes("UTF-8"));
			} catch (IOException e)
			{
				throw new ParamRepositoryException(
						"IO Error while storing param [" + nameParam + "].", e);
			} finally
			{
				try
				{
					if (fileStream != null)
					{
						fileStream.flush();
						fileStream.close();
					}
				} catch (IOException e)
				{
					logger.warn("Unable to close param file: "
							+ paramFile.getAbsolutePath(), e);
				}
			}
		}
	}

	public String get(String name) throws ParamRepositoryException
	{
		FileParamName paramName = asserNameOK(name);

		synchronized (this)
		{
			File paramFile = new File(name);

			if (!paramFile.exists())
			{
				paramFile = toParamFile(paramName);
				if (!paramFile.exists())
				{
					return null;
				}
			}

			InputStream fileStream = null;
			try
			{
				fileStream = new FileInputStream(paramFile);
				byte[] value = StreamUtils.readStream(fileStream);
				return new String(value, "UTF-8");
			} catch (Exception e)
			{
				throw new ParamRepositoryException(
						"Unable to load Param Value from file: "
								+ paramFile.getAbsolutePath(), e);
			} finally
			{
				try
				{
					if (fileStream != null)
					{
						fileStream.close();
					}
				} catch (IOException e)
				{
					logger.warn("Unable to close param file: "
							+ paramFile.getAbsolutePath(), e);
				}
			}
		}
	}

	public void remove(String name)
	{
		FileParamName paramName = asserNameOK(name);

		synchronized (this)
		{
			File paramFile = toParamFile(paramName);
			if (paramFile.exists())
			{
				boolean delResult = paramFile.delete();
			}
		}
	}

	/**
	 * Get the repository root directory.
	 * 
	 * @return The repository root directory.
	 */
	public File getRoot()
	{
		return root;
	}

	/**
	 * Get the param file for the specified param name.
	 * 
	 * @param paramName
	 *            The param name.
	 * @return The param file.
	 */
	public File toParamFile(String name)
	{
		FileParamName paramName = asserNameOK(name);
		return new File(root, paramName.toString());
	}

	/**
	 * Get the param file for the specified param name.
	 * 
	 * @param paramName
	 *            The param name.
	 * @return The param file.
	 */
	private File toParamFile(FileParamName paramName)
	{
		return new File(root, paramName.toString());
	}

	private FileParamName asserNameOK(String name)
	{
		FileParamName paramName;

		if (name == null)
		{
			throw new IllegalArgumentException(
					"null 'name' arg in repository call.");
		}
		try
		{
			paramName = new FileParamName(name);
		} catch (InvalidNameException e)
		{
			throw new IllegalArgumentException("Invalid 'name' arg: " + name, e);
		}
		if (paramName.size() == 0)
		{
			throw new IllegalArgumentException(
					"Invalid 'name' arg.  0 tokens: " + name);
		}

		return paramName;
	}

	/**
	 * Parameter Name. <p/> Represents a compound parameter name - from a
	 * hierarchical namespace.
	 * <h4 id="syntax">Parameter Name Format</h4>
	 * The syntax of the parameter name is expected to be in "directory" format,
	 * read from left to right, where the syntax seperator character is '/' e.g.
	 * "com/acme/ParameterX".
	 * 
	 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
	 */
	private static class FileParamName extends CompoundName
	{

		private static final long serialVersionUID = 1L;

		private static final Properties nameSyntaxProperties = new Properties();
		static
		{
			nameSyntaxProperties.setProperty("jndi.syntax.direction",
					"left_to_right");
			nameSyntaxProperties.setProperty("jndi.syntax.separator", "/");
		};

		/**
		 * Public Constructor.
		 * 
		 * @param name
		 *            The name of the parameter, specified according to the <a
		 *            href="#syntax">defined syntax</a>.
		 * @throws InvalidNameException
		 *             The supplied name violates the <a href="#syntax">defined
		 *             syntax</a>.
		 */
		public FileParamName(String name) throws InvalidNameException
		{
			super(name, nameSyntaxProperties);
		}
	}
}