/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 */

package org.jboss.internal.soa.esb.message.format.xml.marshal;

import java.net.URI;

import org.jboss.internal.soa.esb.message.format.DeferredDeserialisationException;
import org.jboss.soa.esb.UnmarshalException;

/**
 * Wrapper class for marshalled values, supporting JIT deserialisation.
 * 
 * @author <a href='kevin.conner@jboss.com'>Kevin Conner</a>
 */

public class MarshalValueImpl
{
    /**
     * The value.
     */
    private Object value ;
    
    /**
     * The marshalled form.
     */
    private String marshalledForm ;
    /**
     * The plugin type.
     */
    private URI type ;
    
    /**
     * Construct the marshalled value wrapper for the specific value.
     * @param value The serializable value.
     */
    public MarshalValueImpl(final Object value)
    {
        this.value = value ;
    }
    
    /**
     * Construct the marshalled value wrapper for the specific marshalled form.
     * @param type The plugin type.
     * @param marshalledForm The marshalled form.
     */
    public MarshalValueImpl(final URI type, final String marshalledForm)
    {
        this.type = type ;
        this.marshalledForm = marshalledForm ;
    }
    
    /**
     * Get the wrapped value.
     * @return The wrapped value.
     */
    public Object getValue()
    {
        if ((value == null) && (marshalledForm != null))
        {
            final MarshalUnmarshalPlugin plugin = MarshalUnmarshalManager.getInstance().getPlugin(type) ;
            if (plugin == null)
            {
                throw new DeferredDeserialisationException("Could not locate plugin for type: " + type) ;
            }
            try
            {
                value = plugin.unmarshal(marshalledForm) ;
            }
            catch (final UnmarshalException ue)
            {
                throw new DeferredDeserialisationException("Error constructing object value", ue) ;
            }
        }
        marshalledForm = null ;
        return value ;
    }
    
    /**
     * Get the marshalled type.
     * @return The marshalled type.
     */
    URI getMarshalledType()
    {
        return type ;
    }
    
    /**
     * Set the marshalled type.
     * @param type The marshalled type.
     */
    void setMarshalledType(final URI type)
    {
        this.type = type ;
    }
    
    /**
     * Get the marshalled form.
     * @return The marshalled form.
     */
    String getMarshalledForm()
    {
        return marshalledForm ;
    }
    
    /**
     * Set the marshalled form.
     * @return The marshalled form.
     */
    void setMarshalledForm(final String marshalledForm)
    {
        this.marshalledForm = marshalledForm ;
    }
    
    /**
     * Return a string representation of this object.
     * @return the string representation of the value or a deferred identifier.
     */
    public String toString()
    {
        if (value != null)
        {
            return value.toString() ;
        }
        else
        {
            return "Deferred serialized value: " + Integer.toHexString(System.identityHashCode(this)) ;
        }
    }
}
