/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.message.format.xml;

import org.jboss.internal.soa.esb.message.format.MessageSerializer;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.soa.esb.message.Message;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;

/**
 * XML Message serializer.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class XMLMessageSerializer extends MessageSerializer {

    public void serialize(Message message, OutputStream outStream) throws IOException {
        assertMessageInstanceOK(message);

        // Write the preamble.... 
        outStream.write(MessageSerializer.PREAMBLE_XML);

        // Write the message...
        XMLStreamWriter streamWriter;
        try {
            streamWriter = XMLHelper.getXMLStreamWriter(outStream);
        } catch (XMLStreamException e) {
            throw (IOException) (new IOException("Unable to create XMLStreamWriter instance.").initCause(e));
        }

        try {
            String origURI = StreamHelper.writeStartElement(streamWriter, XMLUtil.ESB_QNAME_ENVELOPE);
            ((MessageImpl) message).writeContent(streamWriter) ;
            StreamHelper.writeEndElement(streamWriter, XMLUtil.ESB_QNAME_ENVELOPE.getPrefix(), origURI) ;
            streamWriter.flush();
        } catch (XMLStreamException e) {
            throw (IOException) (new IOException("Error writing message to output stream.").initCause(e));
        } finally {
            try {
                streamWriter.close();
            } catch (XMLStreamException e) {
                throw (IOException) (new IOException("Error closing XMLStreamWriter instance.").initCause(e));
            }
        }
    }

    public Message deserialize(InputStream inStream) throws IOException {
        XMLStreamReader streamReader;

        // Read and test the preamble...
        if(inStream.read() != MessageSerializer.PREAMBLE_XML) {
            throw new IOException("Cannot deserialize message.  Unrecognized message preamble.");
        }

        // Read the message...
        try {
            streamReader = XMLHelper.getXMLStreamReader(new InputStreamReader(inStream)) ;
        } catch (XMLStreamException e) {
            throw (IOException) (new IOException("Unable to create XMLStreamReader instance.").initCause(e));
        }

        try {
            StreamHelper.checkNextStartTag(streamReader, XMLUtil.ESB_QNAME_ENVELOPE) ;
            return new MessageImpl(streamReader);
        } catch (XMLStreamException e) {
            throw (IOException) (new IOException("Error reading message to from stream.").initCause(e));
        } finally {
            try {
                streamReader.close();
            } catch (XMLStreamException e) {
                throw (IOException) (new IOException("Error closing XMLStreamReader instance.").initCause(e));
            }
        }
    }

    private void assertMessageInstanceOK(Message message) throws IOException {
        if (!(isXMLMessage(message))) {
            throw new IOException("Invalid Message instance.  Expecting instance of '" + MessageImpl.class.getName() + "'.");
        }
    }

    public static boolean isXMLMessage(Message message) {
        return message instanceof MessageImpl;
    }
}
