package org.jboss.internal.soa.esb.message.format.xml;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.util.Encoding;
import org.jboss.internal.soa.esb.util.stax.ElementContent;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.internal.soa.esb.util.stax.TextElement;
import org.jboss.soa.esb.message.Context;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * This class is not thread safe and will be duplicated during InVM operations.
 */
public class ContextImpl extends ElementContent implements Context
{
    /**
     * The context variables.
     */
    private Map<String, SerializedValueImpl> context = new HashMap<String, SerializedValueImpl>() ;
    
    /**
     * Default constructor for the context.
     */ 
    public ContextImpl()
    {
    }
    
    /**
     * Create a copy of the context.
     * @param copy The context to copy.
     */
    ContextImpl(final ContextImpl copy)
    {
        context.putAll(copy.context) ;
    }
    
    /**
     * Construct a context from the input stream.
     * 
     * @param in The input stream.
     * @throws XMLStreamException For errors during parsing.
     */
    public ContextImpl(final XMLStreamReader in)
        throws XMLStreamException
    {
        parse(in) ;
    }
    
    public String toString ()
    {
        return "context: " + context ;
    }
    
    /**
     * Set the context value, replacing existing value if present.
     * @param key The context key.
     * @param value The context value.
     * @return the previous value of the context or null if not set.
     */
    public Object setContext(final String key, final Object value)
    {
        AssertArgument.isNotNull(key, "key");
        AssertArgument.isNotNull(value, "value");
        
        if (value instanceof Serializable)
        {
            return unwrap(context.put(key, new SerializedValueImpl((Serializable)value))) ;
        }
        else
        {
            throw new IllegalArgumentException("value must be serializable");
        }
    }
    
    /**
     * Retrieves the context value.
     * @param key The context key.
     * @return The value or null if not present.
     */
    public Object getContext(final String key)
    {
        return unwrap(context.get(key)) ;
    }
    
    /**
     * Remove the context value.
     * @param key The context key.
     * @return The value of the context, or null if not present.
     */
    public Object removeContext(final String key)
    {
        return unwrap(context.remove(key)) ;
    }
    
    /**
     * Get the keys in the context.
     * @return the context keys.
     */
    public Set<String> getContextKeys()
    {
        return Collections.unmodifiableSet(context.keySet()) ;
    }
    
    /**
     * Clear the context.
     */
    public void clearContext()
    {
        context = new HashMap<String, SerializedValueImpl>() ;
    }
    
    /**
     * Write the child content of the element.
     * @param out The output stream.
     * @throws XMLStreamException For errors during output.
     */
    @Override
    protected void writeChildContent(XMLStreamWriter out)
        throws XMLStreamException
    {
        for (Entry<String, SerializedValueImpl> entry: context.entrySet())
        {
            final String origPropertyURI = StreamHelper.writeStartElement(out, XMLUtil.ESB_QNAME_CONTEXT_ENTRY) ;
            
            final TextElement keyElement = new TextElement(Encoding.encodeBytes(entry.getKey().getBytes())) ;
            StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_CONTEXT_ENTRY_KEY, keyElement) ;
            
            final String value = entry.getValue().getSerialisedForm() ;
            final TextElement valueElement = new TextElement(value) ;
            StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_CONTEXT_ENTRY_VALUE, valueElement) ;
            
            StreamHelper.writeEndElement(out, XMLUtil.ESB_QNAME_CONTEXT_ENTRY.getPrefix(), origPropertyURI) ;
        }
    }

    /**
     * Add the element.
     * @param in The current input stream.
     * @param elementName The qualified element name.
     * @throws XMLStreamException For errors during parsing.
     */
    @Override
    protected void putElement(final XMLStreamReader in,
        final QName elementName)
        throws XMLStreamException
    {
        if (XMLUtil.ESB_QNAME_CONTEXT_ENTRY.equals(elementName))
        {
            StreamHelper.checkNextStartTag(in, XMLUtil.ESB_QNAME_CONTEXT_ENTRY_KEY) ;
            final TextElement keyElement = new TextElement(in) ;
            final String key = new String(Encoding.decodeToBytes(keyElement.getText())) ;
            
            StreamHelper.checkNextStartTag(in, XMLUtil.ESB_QNAME_CONTEXT_ENTRY_VALUE) ;
            final TextElement valueElement = new TextElement(in) ;
            final SerializedValueImpl value = new SerializedValueImpl(valueElement.getText()) ;
            
            StreamHelper.checkParentFinished(in) ;
            
            context.put(key, value) ;
        }
        else
        {
            throw new XMLStreamException("Unexpected element name: " + elementName) ;
        }
    }
    
    /**
     * Unwrap the serialised value.
     * @param value The serialised value.
     * @return The unwrapped value.
     */
    private Object unwrap(final SerializedValueImpl value)
    {
        return (value == null ? null : value.getValue()) ;
    }
}