package org.jboss.internal.soa.esb.message.format.serialized;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.util.Util;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/*
 * This implementation requires that all contents are serializable.
 */

public class BodyImpl implements Body, java.io.Serializable
{
	private static final long serialVersionUID = 0x0;

	public BodyImpl()
	{
		_objects = new Hashtable<String, Serializable>();
	}

	public void setByteArray(byte[] content)
	{
		add(BytesBody.BYTES_LOCATION, content);
	}

	public byte[] getByteArray()
	{
		return (byte[]) get(BytesBody.BYTES_LOCATION);
	}

	public void setContents(byte[] content)
	{
		setByteArray(content);
	}

	public byte[] getContents()
	{
		return getByteArray();
	}

	public void add(Object value)
	{
		add(Body.DEFAULT_LOCATION, value);
	}

	public synchronized void add(String name, Object value)
	{
		AssertArgument.isNotNull(value, "value");
		AssertArgument.isNotNull(name, "name");

		if (value instanceof Serializable)
		{
                    _objects.put(name, SerializedValueImpl.wrap((Serializable) value));
		}
		else
			throw new IllegalArgumentException("Object must be Serializable.");
	}

	public Object get()
	{
		return get(Body.DEFAULT_LOCATION);
	}

	public synchronized Object get(String name)
	{
		AssertArgument.isNotNull(name, "name");

                return SerializedValueImpl.unwrap(_objects.get(name));
	}

	public synchronized String[] getNames()
	{
	    final Set<String> keys = _objects.keySet() ;
	    return keys.toArray(new String[0]) ;
	}

        public Enumeration<String> getEnumeratedNames()
        {
            return Collections.enumeration(Arrays.asList(getNames())) ;
        }

	public synchronized Object remove(String name)
	{
            return SerializedValueImpl.unwrap(_objects.remove(name));
	}

	/**
	 * This method is not thread safe.
	 */
	public void replace(Body b)
	{
		if (b == null)
			throw new IllegalArgumentException();

		setByteArray(b.getByteArray());

		_objects = ((BodyImpl) b)._objects;
	}

	/**
	 * This method is not thread safe.
	 */
	public void merge(Body b)
	{
		if (b == null)
			throw new IllegalArgumentException();

		BodyImpl toMerge = null;

		if (b instanceof BodyImpl)
			toMerge = (BodyImpl) b;
		else
			throw new IllegalArgumentException();

		_objects.putAll(toMerge._objects);

		byte[] toAdd = b.getByteArray();
		
		if ((toAdd != null) && (toAdd.length > 0))
			setByteArray(toAdd);
	}

	public String toString()
	{
		StringBuilder toReturn = new StringBuilder("body: [ ");
		byte[] content = getByteArray();
		
		if (content != null)
			toReturn.append("byte[]: ").append(Util.format(new String(content))).append(", ");

		synchronized(this)
		{
			toReturn.append("objects: ").append(_objects.toString());
		}

		return toReturn.append(" ]").toString();
	}

	// copy constructor.

	protected BodyImpl(BodyImpl orig)
	{
		_objects = new Hashtable<String, Serializable>(orig._objects);
	}
	
	protected void initialiseMap(final Map<String, Serializable> payload)
	{
	    synchronized(this)
	    {
	        _objects.clear() ;
	        _objects.putAll(payload) ;
	        SerializedValueImpl.wrapMap(_objects) ;
	    }
	}

	private Hashtable<String, Serializable> _objects;
}