/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.message;

import org.jboss.soa.esb.listeners.message.MessageComposer;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.util.ClassUtil;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * Legacy Message Composer adapter.
 * <p/>
 * The legacy code used a "reflection-only" model for the composers.  We want
 * all such code to implement the {@link MessageComposer} interface, but we also
 * need to be backward compatible.  This class forms an adapter layer around a
 * legacy style composer class, making it look and feel like a {@link MessageComposer}
 * implementation.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class LegacyMessageComposerAdapter<T> implements MessageComposer<T> {

    protected Class _composerClass;

    protected Object _composer;

    protected Method _processMethod;

    protected Method _responderMethod;

    private ConfigTree config;

    public void setConfiguration(ConfigTree config) throws ConfigurationException {
        this.config = config;
        resolveComposerClass();
    }

    public boolean hasResponder() {
        return (_responderMethod != null);
    }
    
    public abstract Class[] getResponderParameters();

    public Message compose(T messagePayload) throws MessageDeliverException {
        try {
            return (Message) _processMethod.invoke(_composer, new Object[] {messagePayload});
        } catch(ClassCastException e) {
            throw new MessageDeliverException("Legacy composer class ('" + _composerClass.getName() + "') process method '" + _processMethod.getName() + "' must return an ESB Message (" + Message.class.getName() + ").");
        } catch (IllegalAccessException e) {
            throw new MessageDeliverException("Legacy composer class ('" + _composerClass.getName() + "') process method '" + _processMethod.getName() + "' is not callable.", e);
        } catch (InvocationTargetException e) {
            throw new MessageDeliverException("Legacy composer class ('" + _composerClass.getName() + "') process method '" + _processMethod.getName() + "' failed with exception.", e.getCause());
        }
    }

    /**
     * Lifted from the old AbstractFileGateway implementation that used reflection
     * for composers.
     *
     * @throws ConfigurationException
     */
    private void resolveComposerClass() throws ConfigurationException {
        String sProcessMethod = null;
        String sResponderMethod = null;
        String composerName = config.getRequiredAttribute(ListenerTagNames.GATEWAY_COMPOSER_CLASS_TAG);

        try {
            _composerClass = ClassUtil.forName(composerName, getClass());
        } catch (ClassNotFoundException e) {
            throw new ConfigurationException("Unable to locate composer class '" + composerName + "' on classpath.");
        }

        Constructor constructor;
        try {
            constructor = _composerClass.getConstructor(new Class[]{ConfigTree.class});
        } catch (NoSuchMethodException e) {
            throw new ConfigurationException("Unable to construct legacy style composer class '" + composerName + "'.  Class must contain a constructor of the form '" + _composerClass.getSimpleName() + "(ConfigTree)'");
        }
        try {
            _composer = constructor.newInstance(config);
        } catch (Exception e) {
            throw new ConfigurationException("Unable to construct legacy style composer class '" + composerName + "'.", e);
        }
        sProcessMethod = config.getAttribute(ListenerTagNames.GATEWAY_COMPOSER_METHOD_TAG, "process");
        sResponderMethod = config.getAttribute(ListenerTagNames.GATEWAY_RESPONDER_METHOD_TAG, "respond");

        try {
            _processMethod = _composerClass.getMethod(sProcessMethod, new Class[]{Object.class});
        } catch (NoSuchMethodException e) {
            throw new ConfigurationException("Legacy style composer class '" + composerName + "' configuration is inavlid.  Class should contain a method called '" + sProcessMethod + "', with a single Object arg.");
        }

        try {
            // So the old stuff seemed to supported variable parameter lists on the responder method.
            // Hard to tell for sure what the intent was because there's only 1 example!!
            _responderMethod = _composerClass.getMethod(sResponderMethod, getResponderParameters());
        } catch (NoSuchMethodException e) {
            throw new ConfigurationException("Legacy style composer class '" + composerName + "' configuration is inavlid.  Class should contain a valid '" + sResponderMethod + "' method.");
        }
    }
}
