/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * (C) 2005-2008, JBoss Inc.
 */
package org.jboss.internal.soa.esb.listeners.war;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.http.HttpConfiguration;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.gateway.http.HttpGatewayServlet;

/**
 * Http Gateway deployment factory.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class HttpGatewayDeploymentFactory implements WebGatewayDeploymentFactory {
    
    public static final String URL_PATTERN = "urlPattern";
    public static final String PROTECTED_METHODS = "protectedMethods";
    public static final String TRANSPORT_GUARANTEE = "transportGuarantee";
    public static final String ALLOWED_ROLES = "allowedRoles";
    public static final String AUTH_METHOD = "authMethod";
    public static final String AUTH_DOMAIN = "authDomain";

    public void updateWebModel(final String deploymentName, ConfigTree webEndpointConfig, WebModel webModel) throws ConfigurationException {
        String servletName = webEndpointConfig.getRequiredAttribute("name");
        Servlet servlet = new Servlet(servletName, HttpGatewayServlet.class.getName(), webModel);
        
        // Set the service associated with the servlet
        servlet.setService( Service.getGatewayTargetService(webEndpointConfig) );
        
        // Reuse the urlMapping for the servlet urlPattern and building the endpoint address
    	String urlMapping;
        String urlPattern = webEndpointConfig.getAttribute(URL_PATTERN);
        if(urlPattern != null) {
            if(urlPattern.startsWith("/")) {
            	urlMapping = "/http" + urlPattern;
            } else {
                urlMapping = "/http/" + urlPattern;
            }
        } else {
            String serviceCat = webEndpointConfig.getAttribute(ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG);
            String serviceName = webEndpointConfig.getAttribute(ListenerTagNames.TARGET_SERVICE_NAME_TAG);
            urlMapping = "/http/" + serviceCat + "/" + serviceName;
        }

        // Set the endpoint urlPattern so as to map requests from that namespace into
        // this servlet instance...
        servlet.getUrlMappings().add(urlMapping);

        String allowedPorts = webEndpointConfig.getAttribute(Servlet.ALLOWED_PORTS);
        if(allowedPorts != null) {
            servlet.getParams().add( new KeyValuePair(Servlet.ALLOWED_PORTS, allowedPorts) );
        }

        // Add the endpoint address of the servlet...
        addEndpointAddress(webEndpointConfig, servlet, urlMapping);

        // Add the security configurations for the servlet...
        addSecurityConstraints(webEndpointConfig, servlet);

        // Map all config tree attributes as servlet parameters...
        servlet.getParams().addAll(webEndpointConfig.attributesAsList());
        servlet.getParams().add(new KeyValuePair(ListenerTagNames.DEPLOYMENT_NAME_TAG, deploymentName)) ;
    }
    
    private void addEndpointAddress(ConfigTree webEndpointConfig, Servlet servlet, String urlMapping) {
        String transportGuarantee = webEndpointConfig.getAttribute(TRANSPORT_GUARANTEE);
        boolean secure = ( "CONFIDENTIAL".equals(transportGuarantee) || "INTEGRAL".equals(transportGuarantee) );
        String path = servlet.getWebModel().getEsbName();
        path = path.substring(0, path.length() - 4) + urlMapping; // remove ".war" and add urlMapping
        while (path.endsWith("*") || path.endsWith("/")) {
        	path = path.substring(0, path.length()-1);
        }
        String address = ( secure ? HttpConfiguration.getSecureAddress(path) : HttpConfiguration.getAddress(path) );
       	servlet.getParams().add( new KeyValuePair(Servlet.ENDPOINT_ADDRESS, address) );
       	servlet.setEndpointAddress(address); // for the deployers
    }

    private void addSecurityConstraints(ConfigTree webEndpointConfig, Servlet servlet) {

        String methodsConfig = webEndpointConfig.getAttribute(PROTECTED_METHODS);
        if(methodsConfig != null) {
            SecurityConstraints securityConstraints = getSecurityConstraints(servlet);
            String[] methods = methodsConfig.split(",");

            // Add the specified methods...
            for(String method : methods) {
                securityConstraints.getProtectedMethods().add(method.trim());
            }
        }

        String rolesConfig = webEndpointConfig.getAttribute(ALLOWED_ROLES);
        if(rolesConfig != null) {
            SecurityConstraints securityConstraints = getSecurityConstraints(servlet);
            String[] roles = rolesConfig.split(",");

            // Add the specified roles...
            for(String role : roles) {
                securityConstraints.getAllowedRoles().add(role.trim());
            }
        }

        String transportGuarantee = webEndpointConfig.getAttribute(TRANSPORT_GUARANTEE);
        if(transportGuarantee != null) {
            getSecurityConstraints(servlet).setTransportGuarantee(transportGuarantee);            
        }
    }

    private SecurityConstraints getSecurityConstraints(Servlet servlet) {
        SecurityConstraints securityConstraints = servlet.getSecurityConstraints();
        if(securityConstraints == null) {
            securityConstraints = new SecurityConstraints(servlet);
        }
        return securityConstraints;
    }
}
