/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.dependencies;

import org.jboss.system.ListenerServiceMBeanSupport;
import org.jboss.system.ServiceMBean;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.logging.Logger;

import javax.management.*;
import java.util.*;

/**
 * Webservice Deployment Barrier Controller Service MBean.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class WSDeploymentBarrierController extends ListenerServiceMBeanSupport implements WSDeploymentBarrierControllerMBean {

    private final static Logger logger = Logger.getLogger(WSDeploymentBarrierController.class);

    private ObjectName wsObjectName;
    private WSDeploymentBarrier wsDeploymentBarrier;

    public ObjectName getWsObjectName() {
        return this.wsObjectName;
    }
    
    public void setWsObjectName(ObjectName wsObjectName) {
        this.wsObjectName = wsObjectName;
    }

    protected void createService() throws Exception {
        List<SubscriptionInfo> subscriptions = new ArrayList<SubscriptionInfo>();
        ObjectName subObjectName = new ObjectName("jboss.web:service=WebServer");

        subscriptions.add(new SubscriptionInfo(subObjectName, "WebServer", null));
        subscribe(subscriptions, true, getServiceName());

        // The WS may already be deployed.  If it is, need to add barrier bean for it...
        registerExistingDeployment();
    }

    protected void destroyService() throws Exception {
        unsubscribe();
        if(wsDeploymentBarrier != null) {
            unregisterBarrierMBean(toBarrierObjectName(wsObjectName));
        }
    }

    private void handle(Notification notification) {
        if(notification instanceof MBeanServerNotification) {
            MBeanServerNotification mbServerNotification = (MBeanServerNotification) notification;

            if(mbServerNotification.getMBeanName().equals(wsObjectName)) {
                ObjectName barrierObjectName;

                try {
                    barrierObjectName = toBarrierObjectName(wsObjectName);
                } catch (MalformedObjectNameException e) {
                    logger.warn("Failed to create WS Deployment Barrier MBean ObjectName for WS deployment '" + wsObjectName + "'.", e);
                    return;
                }

                if(notification.getType().equals(MBeanServerNotification.REGISTRATION_NOTIFICATION)) {
                    registerBarrierMBean(barrierObjectName);
                } else if(notification.getType().equals(MBeanServerNotification.UNREGISTRATION_NOTIFICATION)) {
                    unregisterBarrierMBean(barrierObjectName);
                }
            }
        }
    }

    private void registerBarrierMBean(ObjectName barrierObjectName) {
        if(wsDeploymentBarrier != null) {
            // Already registered.
            return;
        }

        wsDeploymentBarrier = new WSDeploymentBarrier();
        try {
            getServer().registerMBean(wsDeploymentBarrier, barrierObjectName);
            wsDeploymentBarrier.create();
            wsDeploymentBarrier.start();
            logger.debug("Registered WS Barrier: " + barrierObjectName);
        } catch (Exception e) {
            logger.warn("Failed to register WS Deployment Barrier MBean.", e);
        }
    }

    private void unregisterBarrierMBean(ObjectName barrierObjectName) {
        if(wsDeploymentBarrier == null) {
            // Already unregistered.
            return;
        }

        try {
            wsDeploymentBarrier.stop();
            wsDeploymentBarrier.destroy();
            getServer().unregisterMBean(barrierObjectName);
            logger.debug("Unregistered WS Barrier: " + barrierObjectName);
        } catch (Exception e) {
            logger.warn("Failed to unregister WS Deployment Barrier MBean.", e);
        } finally {
            wsDeploymentBarrier = null;
        }
    }

    private void registerExistingDeployment() {
        try {
            Set<?> existingDeployments = getServer().queryMBeans(wsObjectName, null);

            if(existingDeployments != null && !existingDeployments.isEmpty()) {
                // It's already deployed.  Add the Barrier...
                registerBarrierMBean(toBarrierObjectName(wsObjectName));
            }
        } catch (MalformedObjectNameException e) {
            logger.warn("Failed to query existance of WS Barrier Object.", e);
        }
    }

    private static ObjectName toBarrierObjectName(ObjectName wsName) throws MalformedObjectNameException {
        return new ObjectName(wsName.getCanonicalName() + ".Barrier");
    }

    public void handleNotification(Notification notification, Object handback) {
        handle(notification);
        super.handleNotification(notification, handback);
    }

    public void handleNotification2(Notification notification, Object handback) {
        handle(notification);
        super.handleNotification2(notification, handback);
    }

    public static interface WSDeploymentBarrierMBean extends ServiceMBean {
    }
    public static class WSDeploymentBarrier extends ServiceMBeanSupport implements WSDeploymentBarrierMBean {
    }
}