/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.couriers.helpers;

import java.io.IOException;
import java.io.Serializable;

import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.couriers.FaultMessageException;
import org.jboss.soa.esb.listeners.gateway.ESBPropertiesSetter;
import org.jboss.soa.esb.listeners.message.errors.Factory;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.util.Type;
import org.jboss.soa.esb.util.Util;
import org.xml.sax.SAXException;

public class JmsComposer
{
    /**
     * The logger for this class.
     */
    private static final Logger LOGGER = Logger.getLogger(JmsComposer.class) ;
    
    /**
     * Compose a message for the ESB.
     * @param incomingMessage The incoming JMS message.
     * @param propertiesSetter The properties setter.
     * @return The ESB message or null if not a valid ESB message.
     * @throws FaultMessageException if the message is a fault.
     */
    public static Message compose(final Object incomingMessage, final ESBPropertiesSetter propertiesSetter)
        throws FaultMessageException
    {
        if(incomingMessage == null)
        {
            return null ;
        }

        if (!(incomingMessage instanceof ObjectMessage))
        {
            LOGGER.error("Unsupported JMS message type: " + incomingMessage.getClass().getName()+" for incoming ESB-aware message.");
            return null ;
        }
        try
        {
            final ObjectMessage jmsMessage = (ObjectMessage)incomingMessage ;          
            final Serializable obj = (Serializable)jmsMessage.getObject() ;
            final Message composedMessage = Util.deserialize(obj) ;
            
            propertiesSetter.setPropertiesFromJMSMessage(jmsMessage, composedMessage) ;

            /*
                * If this is fault message, then throw an exception with the contents. With the
                * exception of user-defined exceptions, faults will have nothing in the body, properties etc.
                */

            if (Type.isFaultMessage(composedMessage)) {
                Factory.createExceptionFromFault(composedMessage) ;
            }

            return composedMessage ;
        }
        catch (final JMSException jmse)
        {
            LOGGER.error("Failed to read Serialized Object from JMS message.", jmse) ;
            return null ;
        }
        catch (final ClassCastException cce)
        {
            LOGGER.error("Object in JMS message is not a org.jboss.soa.esb.message.Message", cce) ;
        }
        catch (final IOException ioe)
        {
            LOGGER.error("Object in JMS message is not a Serializeable", ioe);
        }
        catch (final ParserConfigurationException pce)
        {
            LOGGER.error("Object in JMS message has invalid XML", pce);
        }
        catch (final SAXException saxe)
        {
            LOGGER.error("Object in JMS message has invalid XML", saxe);
        }
        return null;
    }
}
