/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.enterprise;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;

import javax.management.MBeanServerConnection;

import junit.extensions.TestSetup;
import junit.framework.TestSuite;

/**
 * A test setup that deploys/undeploys archives
 * 
 * @author Thomas.Diesler@jboss.org
 * @since 14-Oct-2004
 */
public class IntegrationTestSetup extends TestSetup {

  private IntegrationTestHelper delegate = new IntegrationTestHelper();
  private String[] archives;
  private ClassLoader originalClassLoader;

  public IntegrationTestSetup(Class testClass, String archiveList) {
    super(new TestSuite(testClass));
    archives = getArchiveArray(archiveList);
  }

  public File getArchiveFile(String archive) {
    return delegate.getTestArchiveFile(archive);
  }

  public URL getArchiveURL(String archive) throws MalformedURLException {
    return delegate.getTestArchiveFile(archive).toURI().toURL();
  }

  public MBeanServerConnection getServer() {
    return delegate.getServer();
  }

  private String[] getArchiveArray(String archiveList) {
    return archiveList == null ? new String[0] : archiveList.split("[\\s,]+");
  }

  protected void setUp() throws Exception {
    super.setUp();

    List clientJars = new ArrayList();
    for (int i = 0; i < archives.length; i++) {
      String archive = archives[i];
      try {
        delegate.deploy(archive);
      }
      catch (Exception ex) {
        ex.printStackTrace();
        delegate.undeploy(archive);
      }

      if (archive.endsWith("-client.jar")) {
        URL archiveURL = getArchiveURL(archive);
        clientJars.add(archiveURL);
      }
    }

    ClassLoader parent = Thread.currentThread().getContextClassLoader();
    originalClassLoader = parent;

    // add client jars to the class loader
    if (!clientJars.isEmpty()) {
      URL[] urls = new URL[clientJars.size()];
      clientJars.toArray(urls);
      URLClassLoader cl = new URLClassLoader(urls, parent);
      Thread.currentThread().setContextClassLoader(cl);
    }
  }

  protected void tearDown() throws Exception {
    try {
      for (int i = 0; i < archives.length; i++) {
        String archive = archives[archives.length - i - 1];
        delegate.undeploy(archive);
      }
    }
    finally {
      Thread.currentThread().setContextClassLoader(originalClassLoader);
    }
    super.tearDown();
  }
}
