/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.instantiation;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;

import org.jbpm.JbpmContext;
import org.jbpm.JbpmException;
import org.jbpm.file.def.FileDefinition;
import org.jbpm.graph.def.ProcessDefinition;

public class ProcessClassLoader extends ClassLoader {

  private ProcessDefinition processDefinition;
  private long processDefinitionId;

  public ProcessClassLoader(ClassLoader parent, ProcessDefinition processDefinition) {
    super(parent);
    // check whether the given process definition is transient
    long id = processDefinition.getId();
    if (id != 0) {
      // persistent, keep id only
      processDefinitionId = id;
    }
    else {
      // transient, keep full object
      this.processDefinition = processDefinition;
    }
  }

  protected ProcessDefinition getProcessDefinition() {
    if (processDefinition != null) return processDefinition;

    JbpmContext jbpmContext = JbpmContext.getCurrentJbpmContext();
    if (jbpmContext != null) {
      return jbpmContext.getGraphSession().loadProcessDefinition(processDefinitionId);
    }

    return null;
  }

  public URL findResource(String name) {
    ProcessDefinition processDefinition = getProcessDefinition();
    FileDefinition fileDefinition;
    if (processDefinition == null
      || (fileDefinition = processDefinition.getFileDefinition()) == null) return null;

    // we know that the leading slashes are removed in the names of the
    // file definitions, therefore we skip the leading slashes
    int off = 0;
    for (int len = name.length(); off < len && name.charAt(off) == '/'; off++)
      /* just increase offset */;

    // if the name of the resources is absolute (starts with one or more slashes)
    if (off > 0) {
      // then start searching from the root of the process archive
      name = name.substring(off);
    }
    else {
      // otherwise (if the resource is relative), look in the classes
      // directory of the file module definition
      name = "classes/" + name;
    }

    byte[] bytes = fileDefinition.getBytes(name);
    if (bytes == null) return null;

    try {
      return new URL("processresource", processDefinition.getName(), -1, "classes/" + name,
        new BytesUrlStreamHandler(bytes));
    }
    catch (MalformedURLException e) {
      throw new JbpmException("could not create url", e);
    }
  }

  public static class BytesUrlStreamHandler extends URLStreamHandler {

    private byte[] bytes;

    public BytesUrlStreamHandler(byte[] bytes) {
      this.bytes = bytes;
    }

    protected URLConnection openConnection(URL u) throws IOException {
      return new BytesUrlConnection(bytes, u);
    }
  }

  public static class BytesUrlConnection extends URLConnection {

    private byte[] bytes;

    public BytesUrlConnection(byte[] bytes, URL u) {
      super(u);
      this.bytes = bytes;
    }

    public void connect() throws IOException {
    }

    public InputStream getInputStream() throws IOException {
      return new ByteArrayInputStream(bytes);
    }
  }

  public Class findClass(String className) throws ClassNotFoundException {
    ProcessDefinition processDefinition = getProcessDefinition();
    FileDefinition fileDefinition;
    if (processDefinition == null
      || (fileDefinition = processDefinition.getFileDefinition()) == null) {
      throw new ClassNotFoundException(className);
    }

    // look in the classes directory of the file module definition
    String fileName = "classes/" + className.replace('.', '/') + ".class";
    byte[] classBytes = fileDefinition.getBytes(fileName);
    if (classBytes == null) throw new ClassNotFoundException(className);

    // define the package before defining the class
    // see https://jira.jboss.org/jira/browse/JBPM-1404
    int packageIndex = className.lastIndexOf('.');

    if (packageIndex != -1) {
      String packageName = className.substring(0, packageIndex);

      if (getPackage(packageName) == null) {
        Package jbpmPackage = ProcessClassLoader.class.getPackage();
        definePackage(packageName, null, null, null, jbpmPackage.getImplementationTitle(),
          jbpmPackage.getImplementationVersion(), jbpmPackage.getImplementationVendor(), null);
      }
    }
    return defineClass(className, classBytes, 0, classBytes.length);
  }
}
