/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.server.ebws;

import java.util.Arrays;
import java.util.Set;
import java.util.TreeSet;

import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;

import org.jboss.mx.util.MBeanProxyExt;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.soa.esb.actions.AbstractActionLifecycle;
import org.jboss.soa.esb.actions.ActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.Properties;

/**
 * Action class to attempt a suspend of a running jBPM process.
 * 
 * @author <a href='mailto:Kevin.Conner@jboss.com'>Kevin Conner</a>
 */
public class LogMessageAction extends AbstractActionLifecycle implements ActionPipelineProcessor
{
    private final String requestLocation ;
    private final String responseLocation ;
    
    public LogMessageAction(final ConfigTree config)
    {
        final ConfigTree parent = config.getParent() ;
        requestLocation = parent.getAttribute(ListenerTagNames.REQUEST_LOCATION_TAG) ;
        responseLocation = parent.getAttribute(ListenerTagNames.RESPONSE_LOCATION_TAG) ;
    }

    public Message process(final Message message)
        throws ActionProcessingException
    {
        final MBeanServer server = MBeanServerLocator.locateJBoss();
        
        final EBWSServerMBean ebwsServer ;
        try
        {
            ebwsServer = (EBWSServerMBean) MBeanProxyExt.create(EBWSServerMBean.class, EBWSServerMBean.OBJECT_NAME, server);
        }
        catch (final MalformedObjectNameException mone)
        {
            throw new ActionProcessingException("Error creating MBean proxy", mone) ;
        }
        
        final Body body = message.getBody() ;
        final Object messageRequest ;
        if (requestLocation != null)
        {
            messageRequest = body.get(requestLocation) ;
        }
        else
        {
            messageRequest = body.get() ;
        }
        final String contents = String.valueOf(messageRequest) ;
        ebwsServer.logMessage(contents) ;
        
        final Properties props = message.getProperties() ;
        final String wsaMessageID = (String) props.getProperty(Environment.WSA_MESSAGE_ID) ;
        log(ebwsServer, wsaMessageID) ;
        
        final String[] relatesTo = (String[]) props.getProperty(Environment.WSA_RELATES_TO) ;
        log(ebwsServer, relatesTo) ;
        
        final String[] relationshipType = (String[]) props.getProperty(Environment.WSA_RELATIONSHIP_TYPE) ;
        log(ebwsServer, relationshipType) ;
        
        final String response = (message.getBody().get(EBWSUtil.ERROR) != null ? EBWSUtil.INVALID_RESPONSE : EBWSUtil.VALID_RESPONSE) ;
        ebwsServer.logMessage(response) ;
        
        if (responseLocation != null)
        {
            body.add(responseLocation, response) ;
        }
        else
        {
            body.add(response) ;
        }
        return message;
    }

    private static void log(final EBWSServerMBean server, final String message)
    {
        if (message != null)
        {
            server.logMessage(message) ;
        }
    }

    private static void log(final EBWSServerMBean server, final String[] messages)
    {
        final int numMessages = (messages == null ? 0 : messages.length) ;
        if (numMessages > 0)
        {
            final Set<String> messageSet = new TreeSet<String>(Arrays.asList(messages)) ;
            log(server, messageSet.toString()) ;
        }
    }

    public void processException(final Message message, final Throwable th)
    {
    }
    
    public void processSuccess(final Message message)
    {
    }
}
