/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.soap;

import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.soap.SOAPException;
import javax.xml.ws.WebServiceException;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPHandler;
import javax.xml.ws.handler.soap.SOAPMessageContext;

import org.picketlink.identity.federation.core.exceptions.ProcessingException;
import org.picketlink.identity.federation.core.wstrust.SamlCredential;
import org.jboss.soa.esb.services.security.auth.login.SamlContext;
import org.w3c.dom.Element;

/**
 * SOAPSamlHandler is a SOAP Protocol Handler that will add a security header to
 * all outgoing messages if a SamlPrincipal has been set using the
 * {@link SamlContext}.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class SOAPSamlHandler implements SOAPHandler<SOAPMessageContext>
{
    public final static String WSSE_NS = "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd";
    QName securityQName = new QName(SOAPSamlHandler.WSSE_NS, "Security");

    public boolean handleMessage(final SOAPMessageContext soapContext)
    {
        final Boolean outBound = (Boolean) soapContext.get(MessageContext.MESSAGE_OUTBOUND_PROPERTY);
        if (outBound.booleanValue())
            return true;

        try
        {
            final SamlCredential samlCredential = SamlContext.getFirstSamlCredential();
            if (samlCredential != null)
            {
                final Element assertionElement = samlCredential.getAssertionAsElement();
	            SOAPSamlHandlerUtil.addAssertion(soapContext, securityQName, assertionElement);
            }
        }
        catch (final SOAPException e)
        {
            throw new WebServiceException(e.getMessage(), e);
        }
        catch (ProcessingException e)
        {
            throw new WebServiceException(e.getMessage(), e);
        }
        return true;
    }

    public Set<QName> getHeaders()
    {
        return null;
    }

    public void close(final MessageContext messageContext)
    {
    }

    public boolean handleFault(final SOAPMessageContext soapContext)
    {
        return false;
    }

}
