/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.Properties;

import javax.naming.Context;

import junit.framework.JUnit4TestAdapter;

import org.jboss.security.auth.callback.AppCallbackHandler;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Test;

/**
 * Unittest for {@link EJBProcessor}
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 */
public class EJBProcessorUnitTest
{
	final String payloadLocation = "location";
	final String payload = "Hello, hello Clarice";
	final String outputLocation = "output";

	@Test
	public void processEjb3() throws ActionLifecycleException, ActionProcessingException, ConfigurationException
	{
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add(payloadLocation, payload);
		
		final ConfigBuilder builder = new ConfigBuilder();
		builder.methodName("printMessage");
		builder.outputLocation(outputLocation);
		builder.jndiName("Simple");
		builder.isEjb3(true);
		builder.initCtxFactory("org.jnp.interfaces.NamingContextFactory");
		builder.providerUrl("jnp://localhost:1099");
		builder.arg(String.class, payloadLocation);
		
		final ConfigTree config = builder.build();
		
		EJBProcessor processor = new MockEjbProcessor(config);
		processor.initialise();

		Message process = processor.process(message);
		assertNull( process.getBody().get(outputLocation));
	}

	@Test
	public void processEjb3WithReturnValue() throws ActionLifecycleException, ActionProcessingException, ConfigurationException
	{
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add(payloadLocation, payload);
		
		final ConfigBuilder builder = new ConfigBuilder();
		builder.methodName("printMessageAndReturn");
		builder.outputLocation(outputLocation);
		builder.jndiName("Simple");
		builder.isEjb3(true);
		builder.initCtxFactory("org.jnp.interfaces.NamingContextFactory");
		builder.providerUrl("jnp://localhost:1099");
		builder.arg(String.class, payloadLocation);
		final ConfigTree config = builder.build();

		EJBProcessor processor = new MockEjbProcessor(config);
		processor.initialise();

		Message process = processor.process(message);
		Object object = process.getBody().get(outputLocation);
		assertTrue( object instanceof String);
		assertEquals( payload, object );
	}

	@Test (expected = ActionLifecycleException.class )
	public void initWithNullMethodName() throws ActionLifecycleException, ActionProcessingException, ConfigurationException
	{
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add(payloadLocation, payload);
		
		final ConfigBuilder builder = new ConfigBuilder();
        builder.outputLocation(outputLocation);
        builder.jndiName("Simple");
        builder.isEjb3(true);
        builder.initCtxFactory("org.jnp.interfaces.NamingContextFactory");
        builder.providerUrl("jnp://localhost:1099");
        builder.arg(String.class, payloadLocation);
        final ConfigTree config = builder.build();

		EJBProcessor processor = new MockEjbProcessor(config);
		processor.initialise();
	}

	@Test
	public void nullPayload() throws ActionLifecycleException, ActionProcessingException, ConfigurationException
	{
		Message message = MessageFactory.getInstance().getMessage();
		// Not adding a message body
		
		final ConfigBuilder builder = new ConfigBuilder();
        builder.methodName("printMessage");
        builder.outputLocation(outputLocation);
        builder.jndiName("Simple");
        builder.isEjb3(true);
        builder.initCtxFactory("org.jnp.interfaces.NamingContextFactory");
        builder.providerUrl("jnp://localhost:1099");
        builder.arg(String.class, null);
        final ConfigTree config = builder.build();

		EJBProcessor processor = new MockEjbProcessor(config);
		processor.initialise();
		Message process = processor.process(message);
		Object object = process.getBody().get(outputLocation);
		assertNull(object);
	}
	
	@Test
    public void verifySecurityConfig() throws ActionLifecycleException, ActionProcessingException, ConfigurationException
    {
        Message message = MessageFactory.getInstance().getMessage();
        message.getBody().add(payloadLocation, payload);
        
        final ConfigBuilder builder = new ConfigBuilder();
        builder.methodName("printMessageAndReturn");
        builder.outputLocation(outputLocation);
        builder.jndiName("Simple");
        builder.isEjb3(true);
        builder.initCtxFactory("org.jnp.interfaces.NamingContextFactory");
        builder.providerUrl("jnp://localhost:1099");
        builder.arg(String.class, payloadLocation);
        builder.username("kalle");
        builder.password("12345");
        final ConfigTree config = builder.build();

        MockEjbProcessor processor = new MockEjbProcessor(config);
        processor.initialise();
        
        Properties jndiProperties = processor.getJndiProperties();
        assertTrue(jndiProperties.containsKey(Context.SECURITY_PRINCIPAL));
        assertTrue(jndiProperties.containsKey(Context.SECURITY_CREDENTIALS));
        assertEquals("kalle", jndiProperties.get(Context.SECURITY_PRINCIPAL));
        assertEquals("12345", jndiProperties.get(Context.SECURITY_CREDENTIALS));
    }
	
	@Test (expected = ActionLifecycleException.class)
    public void verifyMissingSecurityCredentials() throws ActionLifecycleException, ActionProcessingException, ConfigurationException
    {
        Message message = MessageFactory.getInstance().getMessage();
        message.getBody().add(payloadLocation, payload);
        
        final ConfigBuilder builder = new ConfigBuilder();
        builder.methodName("printMessageAndReturn");
        builder.outputLocation(outputLocation);
        builder.jndiName("Simple");
        builder.isEjb3(true);
        builder.initCtxFactory("org.jnp.interfaces.NamingContextFactory");
        builder.providerUrl("jnp://localhost:1099");
        builder.arg(String.class, payloadLocation);
        builder.username("kalle");
        final ConfigTree config = builder.build();

        MockEjbProcessor processor = new MockEjbProcessor(config);
        processor.initialise();
    }
	
	@Test
    public void verifySecurityLoginModuleConfig() throws ActionLifecycleException, ActionProcessingException, ConfigurationException
    {
        Message message = MessageFactory.getInstance().getMessage();
        message.getBody().add(payloadLocation, payload);
        
        final ConfigBuilder builder = new ConfigBuilder();
        builder.methodName("printMessageAndReturn");
        builder.outputLocation(outputLocation);
        builder.jndiName("Simple");
        builder.isEjb3(true);
        builder.initCtxFactory("org.jnp.interfaces.NamingContextFactory");
        builder.providerUrl("jnp://localhost:1099");
        builder.arg(String.class, payloadLocation);
        builder.username("kalle");
        builder.password("12345");
        builder.securityLoginModule("SampleModule");
        final ConfigTree config = builder.build();

        MockEjbProcessor processor = new MockEjbProcessor(config);
        processor.initialise();
        
        assertEquals("SampleModule", processor.getLoginModuleName());
        assertTrue(processor.getCallbackHandler() instanceof AppCallbackHandler);
    }

	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter(EJBProcessorUnitTest.class);
	}

}
