/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.routing.cbr;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.math.BigDecimal;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.actions.CBRConfigTreeBuilder;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.message.mapping.ObjectMapper;
import org.jboss.soa.esb.message.mapping.ObjectMappingException;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.jboss.soa.esb.services.routing.cbr.ContentBasedRouterFactory;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Tests for {@link JBossRulesRouter}.
 *
 * @author kurt.stam@redhat.com
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class JBossRulesRouterUnitTest
{
    //	instance under test
	private JBossRulesRouter jbrRouter;

	@Test
	public void routeSerializedMessage() throws MessageRouterException
	{
		Message message = createMessage( MessageType.JAVA_SERIALIZED );
		ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBRules.drl").build();
		jbrRouter.setConfigTree( configTree );

		List<String> destinationServices = jbrRouter.route(null, false, message, null);
		assertNotNull(destinationServices);
		assertTrue(destinationServices.size()>0);
		assertEquals(destinationServices.iterator().next(),"serialized-destination");
	}

	@Test
	public void routeSerializedMessageDecisionTableStateless() throws MessageRouterException
	{
		Message message = createMessage( MessageType.JAVA_SERIALIZED );

		ConfigTree configTree = new CBRConfigTreeBuilder( true ).decisionTable( "RuleBaseHelper.xls").build();
		jbrRouter.setConfigTree( configTree );

		List<String> destinationServices = jbrRouter.route( null, false, message, null);

		assertNotNull( destinationServices );
		assertTrue( "One destination should have been added by Drools", destinationServices.size() == 1 );
		assertEquals( "serialized-destination", destinationServices.get(0).toString() );
	}

	@Test
	public void routeSerializedMessageDecisionTableStateful() throws MessageRouterException
	{
		Message message = createMessage( MessageType.JAVA_SERIALIZED );

		ConfigTree configTree = new CBRConfigTreeBuilder( true ).decisionTable( "RuleBaseHelper.xls").stateful( true ).build();
		jbrRouter.setConfigTree( configTree );

		List<String> destinationServices = jbrRouter.route( null, false, message, null);

		assertNotNull( destinationServices );
		assertTrue( "One destination should have been added by Drools", destinationServices.size() == 1 );
		assertEquals( "serialized-destination", destinationServices.get(0).toString() );
	}

	@Test
	public void routeXMLMessage() throws MessageRouterException
	{
		Message message = createMessage( MessageType.JBOSS_XML);
		ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBRules.drl").build();
		jbrRouter.setConfigTree( configTree );
		List<String> destinationServices = jbrRouter.route(null, null, false, message, null);
		assertEquals(destinationServices.iterator().next(),"xml-destination");
	}

	@Test
	public void routeXMLMessageUsingXPathMatch() throws MessageRouterException
	{
		Message message = createMessage( MessageType.JBOSS_XML);
		message.getBody().add(("<jbossesb>TEST BODY</jbossesb>").getBytes());
		ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBRules-XPath.drl").ruleLanguage("XPathLanguage.dsl").build();
		jbrRouter.setConfigTree( configTree );

		List<String> destinationServices = jbrRouter.route(null,null, false, message, null);
		assertEquals(destinationServices.iterator().next(),"XML_XPath_Destination");
	}

	@Test
	public void routeXMLMessageUsingXPathEquals() throws MessageRouterException
	{
		Message message = createMessage( MessageType.JBOSS_XML );
		message.getBody().add(("<Dave>rocks</Dave>").getBytes());

		ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBRules-XPath.drl").ruleLanguage("XPathLanguage.dsl").build();
		jbrRouter.setConfigTree( configTree );

		List<String> destinationServices = jbrRouter.route(null,null, false, message, null);
		assertEquals(destinationServices.iterator().next(),"XML_XPath_Dave_Destination");
	}

	@Test
	public void routeXMLMessageUsingXPathGreaterThen() throws MessageRouterException
	{
		Message message = createMessage( MessageType.JBOSS_XML );
		message.getBody().add(("<price>1.55</price>").getBytes());

		ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBRules-XPath.drl").ruleLanguage("XPathLanguage.dsl").build();
		jbrRouter.setConfigTree( configTree );

		List<String> destinationServices = jbrRouter.route(null, null,false, message, null);
		assertEquals(destinationServices.iterator().next(),"XML_XPath_GreaterThan_Destination");
	}

	@Test
	public void routeXMLMessageUsingXPathLessThen()
	{
		try {
			//add new messages
			Message message = MessageFactory.getInstance().getMessage( MessageType.JBOSS_XML );
			//set the body inside the Message
			message.getBody().add(("<price>0.55</price>").getBytes());
			ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBRules-XPath.drl").ruleLanguage("XPathLanguage.dsl").build();
    		jbrRouter.setConfigTree( configTree );

			List<String> destinationServices = jbrRouter.route(null, null,false,message,null);
			assertEquals(destinationServices.iterator().next(),"XML_XPath_LessThan_Destination");
		} catch (MessageRouterException e) {
			e.printStackTrace();
		}
	}

	@Test ( expected = IllegalArgumentException.class )
	public void shouldThrowIfNoRuleSetIsSupplied() throws MessageRouterException
	{
		//add new messages
		Message message = MessageFactory.getInstance().getMessage( MessageType.JBOSS_XML );
		ConfigTree configTree = new ConfigTree("bad-config");
		jbrRouter.setConfigTree( configTree );

		jbrRouter.route( null ,"XPathLanguage.dsl",false,message,null);
	}

	@Test ( expected = IllegalStateException.class )
	public void shouldThrowIfRuleServiceImplIsInvalid() throws MessageRouterException
	{
		//add new messages
		Message message = MessageFactory.getInstance().getMessage( MessageType.JBOSS_XML );
		ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleServiceImpl( "bad.Class" ).ruleFile( "JBossESBRules.drl").build();
		jbrRouter.setConfigTree( configTree );

		jbrRouter.route( null, "XPathLanguage.dsl", false, message, null);
	}

	//	Moved into this class from BusinessProcessRoutingUnitTest
	//	as that class also used JBossRulesRouter as you can see below
	@Test
	public void discount() throws ObjectMappingException
	{
        //new messages
		Message message = createMessage( MessageType.JAVA_SERIALIZED );
		Order order = new Order();
        order.setQuantity(20);
        order.setUnitPrice(new BigDecimal("20.0"));
		message.getBody().add("Order", order);
        //extract the order from the message, so the drl can process it
        List<String> messagePathList = new ArrayList<String>();
        messagePathList.add("body.Order");
        ObjectMapper mapper = new ObjectMapper();
        List<Object> objectList = mapper.createObjectList(message, messagePathList);
        //Now send to the rules engine
		JBossRulesRouter jbossRulesRouter = new JBossRulesRouter();
		ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBPricingRoutingRules.drl").build();
		jbossRulesRouter.setConfigTree( configTree );
        try {
    		List<String> destinations = jbossRulesRouter.route("JBossESBPricingRoutingRules.drl", false, message, objectList);
            assertEquals(order.getDiscount(),10.0, 0);
            assertEquals("10%",message.getBody().get("DiscountObject"));
            String shippingDestination = destinations.iterator().next();
            System.out.println(shippingDestination);
            assertEquals("express-shipping-destination", shippingDestination);

        } catch (MessageRouterException mre) {
            System.out.println("Exception was thrown.");
            mre.printStackTrace();
            assertTrue(false);
        }
	}

	//	Moved into this class from BusinessProcessRoutingUnitTest
	//	as that class also used JBossRulesRouter as you can see befinal low
    @Test
    public void nodiscount() throws ObjectMappingException
    {
        //new messages
        Message message = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
        Order order = new Order();
        order.setQuantity(2);
        order.setUnitPrice(new BigDecimal("20.0"));
        message.getBody().add("Order", order);
//      extract the order from the message, so the drl can process it
        List<String> messagePathList = new ArrayList<String>();
        messagePathList.add("body.Order");
        ObjectMapper mapper = new ObjectMapper();
        List<Object> objectList = mapper.createObjectList(message, messagePathList);
        //Now send to the rules engine
        JBossRulesRouter jbossRulesRouter = new JBossRulesRouter();
		ConfigTree configTree = new CBRConfigTreeBuilder( true ).ruleFile( "JBossESBPricingRoutingRules.drl").build();
		jbossRulesRouter.setConfigTree( configTree );
        try {
            List<String> destinations = jbossRulesRouter.route("JBossESBPricingRoutingRules.drl", false, message, objectList);
            assertEquals(order.getDiscount(),0.0, 0);
            assertEquals("0%",message.getBody().get("DiscountObject"));
            String shippingDestination = destinations.iterator().next();
            System.out.println(shippingDestination);
            assertEquals("normal-shipping-destination", shippingDestination);

        } catch (MessageRouterException mre) {
            System.out.println("Exception was thrown.");
            mre.printStackTrace();
            assertTrue(false);
        }
    }

	@Before
	public void setup() throws MessageRouterException
	{
		jbrRouter = (JBossRulesRouter) ContentBasedRouterFactory.getRouter(org.jboss.soa.esb.actions.ContentBasedRouter.DEFAULT_CBR_CLASS);
		jbrRouter.setConfigTree( new ConfigTree("dummy" ));
	}

	@BeforeClass
	public static void runBeforeAllTests() throws Exception
	{
		try {
			TestEnvironmentUtil.setESBPropertiesFileToUse();
		} catch (Exception e) {
			e.printStackTrace();
			System.out.println("We should stop testing, since we don't any config properties");
			assertTrue(false);
		}
	}

	private Message createMessage( final URI type )
	{
		//new messages
		Message message = MessageFactory.getInstance().getMessage( type );
		//set some properties inside the Message
		message.getProperties().setProperty("prop1", "val1");
		message.getProperties().setProperty("prop2", "val2");
		//set the body inside the Message
		message.getBody().add(("TEST BODY").getBytes());
		//set some object attachments inside the Message
		message.getAttachment().addItem(new String("TEST ATTACHMENT1"));
		message.getAttachment().addItem(new String("TEST ATTACHMENT2"));
		return message;
	}

	public static junit.framework.Test suite() {
		return new JUnit4TestAdapter(JBossRulesRouterUnitTest.class);
	}

}
