/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.rules;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.log4j.Logger;
import org.drools.RuleBase;
import org.drools.StatefulSession;
import org.drools.agent.RuleAgent;
import org.drools.compiler.DroolsParserException;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.lifecycle.LifecyclePriorities;
import org.jboss.soa.esb.lifecycle.LifecycleResource;
import org.jboss.soa.esb.lifecycle.LifecycleResourceException;
import org.jboss.soa.esb.lifecycle.LifecycleResourceFactory;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.rules.RuleInfo;
import org.jboss.soa.esb.services.rules.RuleService;
import org.jboss.soa.esb.services.rules.StatefulRuleInfo;

/**
 * JBossRules (aka Drools) Implementation of a rule engine interface for rules services. Here we use
 * <p/>
 *
 * @author jdelong@redhat.com
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class DroolsRuleService implements RuleService
{
    private static Logger log = Logger.getLogger(DroolsRuleService.class);

    /**
	 * The lifecycle resource state factory for RuleBases.
	 */
	private static final LifecycleResourceFactory<Map<String, DroolsRuleBaseState>> lifecycleRuleBaseStateFactory = new LifecycleRuleBaseStateFactory();

    /**
	 * The lifecycle resource factory for RuleAgents.
	 */
	private static final LifecycleResourceFactory<Map<String, RuleAgent>> lifecycleRuleAgentFactory = new LifecycleRuleAgentFactory();

	/**
	 * The lifecycle resource factory rule sets.
	 */
	private static final LifecycleResourceFactory<ConcurrentHashMap<String, String>> lifecycleRuleSetFactory = new LifecycleRuleSetFactory();

	/**
	 * The lifecycle resource rule base states.
	 */
	private static final LifecycleResource<Map<String, DroolsRuleBaseState>> lifecycleRuleBaseStates = new LifecycleResource<Map<String, DroolsRuleBaseState>>(
			lifecycleRuleBaseStateFactory, LifecyclePriorities.RULE_BASE_PRIORITY);

	/**
	 * RuleAgents cache
	 */
	private static final LifecycleResource<Map<String, RuleAgent>> lifecycleRuleAgents = new LifecycleResource<Map<String, RuleAgent>>(
			lifecycleRuleAgentFactory, LifecyclePriorities.RULE_BASE_PRIORITY);

	/**
	 * The lifecycle resource for ruleset.
	 */
	private static final LifecycleResource<ConcurrentHashMap<String, String>> lifecycleRuleSets = new LifecycleResource<ConcurrentHashMap<String, String>>(
			lifecycleRuleSetFactory, LifecyclePriorities.RULE_BASE_PRIORITY);

	/**
	 * Execute rules using a certain ruleSet and domain specific language using
	 * the Stateless rule engine API
	 *
	 * @param ruleSet -
	 *            String reference to a file which contains a ruleSet.
	 * @param dsl -
	 *            String reference to a file which contains a custom rule
	 *            language definition
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading
	 *            the ruleSet.
	 * @param message -
	 *            Message that is updated with the results.
	 * @param globals -
	 *            Map of globals variables that should be set in the working memory
	 * @param objectList -
	 *            a list with additional objects (typically pulled from the
	 *            message) to be inserted into working memory
	 *
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #executeStatelessRules(RuleInfo, Message)} instead.
	 */
	@Deprecated
    public Message executeStatelessRules(
			final String ruleSet,
			final String dsl,
			final boolean ruleReload,
			Message message,
			final Map<String,Object> globals,
			final List<Object> objectList) throws RuleServiceException
	{
		AssertArgument.isNotNullAndNotEmpty( ruleSet, "ruleSet" );

		DroolsRuleBaseState ruleBaseState = getRuleBaseStateForFileBasedRules(ruleSet, dsl, ruleReload);
		return ruleBaseState.executeStatelessRules(message, globals, objectList);
	}

	/**
	 * Execute rules using the Stateless rule engine API.
	 *
	 * @param ruleInfo The {@link RuleInfo} object contain processing instructions for the rule engine. Must not be null.
	 * @param message The ESB Message object.
	 *
	 */
	public Message executeStatelessRules(final RuleInfo ruleInfo, final Message message) throws RuleServiceException
	{
		AssertArgument.isNotNull(ruleInfo, "ruleInfo" );
		final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForFileBasedRules(ruleInfo.getRuleSource(), ruleInfo.getDslSource(), ruleInfo.getReload());
		return ruleBaseState.executeStatelessRules(message, ruleInfo.getGlobals(), ruleInfo.getDefaultFacts());
	}


	/**
	 * Execute rules from a decision table using the Stateless rule engine API
	 *
	 * @param decisionTable -
	 *            String reference to a file which contains a spreadsheet of rules
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading
	 *            the ruleSet.
	 * @param message -
	 *            Message that is updated with the results.
	 * @param globals -
	 *            Map of globals variables that should be set in the working memory
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into
	 *            working memory
	 *
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #executeStatelessRulesFromDecisionTable(RuleInfo, Message)} instead.
	 */
	@Deprecated
    public Message executeStatelessRulesFromDecisionTable(
			final String decisionTable,
			final boolean ruleReload,
			Message message,
			final Map<String,Object> globals,
			final List<Object> objectList) throws RuleServiceException
	{
		AssertArgument.isNotNullAndNotEmpty( decisionTable, "decisionTable" );

		DroolsRuleBaseState ruleBaseState = getRuleBaseStateForDecisionTable( decisionTable, ruleReload );
		return ruleBaseState.executeStatelessRules( message, globals, objectList );
	}

	public Message executeStatelessRulesFromDecisionTable(final RuleInfo ruleInfo, final Message message) throws RuleServiceException
	{
		AssertArgument.isNotNull(ruleInfo, "ruleInfo" );
		DroolsRuleBaseState ruleBaseState = getRuleBaseStateForDecisionTable(ruleInfo.getRuleSource(), ruleInfo.getReload());
		return ruleBaseState.executeStatelessRules(message, ruleInfo.getGlobals(), ruleInfo.getDefaultFacts());

	}

	/**
	 * Execute rules using a rule package retrieved via the Rule Agent from a URL of local file system
	 *            using the Stateless rule engine API
	 *
	 * @param ruleAgentProperties -
	 *            String reference to a file which contains properties used by the RuleAgent to locate a rule package.
	 * @param message -
	 *            Message that is updated with the results.
	 * @param globals -
	 *            Map of globals variables that should be set in the working memory
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into
	 *            working memory
	 *
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #executeStatelessRulesFromRuleAgent(RuleInfo, Message)} instead.
	 */
	@Deprecated
    public Message executeStatelessRulesFromRuleAgent(
			final String ruleAgentProperties,
			Message message,
			final Map<String,Object> globals,
			final List<Object> objectList) throws RuleServiceException
	{
		AssertArgument.isNotNullAndNotEmpty( ruleAgentProperties, "ruleAgentProperties" );

		try
		{
    		final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForRuleAgent( ruleAgentProperties ) ;
			return ruleBaseState.executeStatelessRules(message, globals, objectList);
		}
		catch ( final IOException e)
		{
			throw new RuleServiceException( "Could not read the ruleAgentProperties file [" + ruleAgentProperties + "]", e);
		}
		catch ( final Exception e)
		{
			throw new RuleServiceException( "RuleAgent could not get the RuleBase. " + e.getMessage(), e);
		}
	}

	public Message executeStatelessRulesFromRuleAgent(final RuleInfo ruleInfo, final Message message) throws RuleServiceException
	{
		AssertArgument.isNotNull(ruleInfo, "ruleInfo" );
		try
        {
            final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForRuleAgent(ruleInfo.getRuleSource());
            return ruleBaseState.executeStatelessRules(message, ruleInfo.getGlobals(), ruleInfo.getDefaultFacts());
        }
        catch ( final IOException e)
        {
            throw new RuleServiceException( "Could not read the ruleAgentProperties file [" + ruleInfo.getRuleSource() + "]", e);
        }
        catch ( final Exception e)
        {
            throw new RuleServiceException( "RuleAgent could not get the RuleBase. " + e.getMessage(), e);
        }

	}

	/**
	 * Execute rules using a certain ruleSet and domain specific language using the Stateful rule engine API
	 *
	 * @param ruleSet -
	 *            String reference to a file which contains a ruleSet.
	 * @param dsl -
	 *            String reference to a file which contains a custom rule
	 *            language definition(Domain Specific Language)
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading
	 *            the ruleSet.
	 * @param message -
	 *            Message that is updated with the results.
	 * @param globals -
	 *            Map of globals variables that should be set in the working memory
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into
	 *            working memory
	 *
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #executeStatefulRules(RuleInfo, Message)} instead.
	 */
	@Deprecated
    public Message executeStatefulRules(
			final String ruleSet,
			final String dsl,
			final boolean ruleReload,
			Message message,
			final Map<String,Object> globals,
			final List<Object> objectList) throws RuleServiceException
	{
		AssertArgument.isNotNullAndNotEmpty( ruleSet, "ruleSet" );

		final RuleInfoBuilder builder = new RuleInfoBuilder(ruleSet);
		builder.dslSource(dsl);
		builder.reload(ruleReload);
		builder.globals(globals);
		builder.defaultFacts(objectList);
		StatefulRuleInfoImpl ruleInfo = new StatefulRuleInfoImpl(builder.build(), false, false);

		DroolsRuleBaseState ruleBaseState = getRuleBaseStateForFileBasedRules( ruleInfo.getRuleSource(), ruleInfo.getDslSource(), ruleInfo.getReload());
		return ruleBaseState.executeStatefulRules(ruleInfo, message);
	}

	/**
	 * Execute rules using a certain ruleSet and domain specific language using the Stateful rule engine API
	 *
	 * @param info {@link StatefulRuleInfo} containing the execution information for the Rule engine.
	 * @param msg The ESB Message object.
	 * @return {@link Message} The ESB Message Object.
	 *
	 * @throws RuleServiceException If a problem occurs while trying to create the Drools RuleBase or if an exception
	 *                             occurs during execution
	 */
	public Message executeStatefulRules(final StatefulRuleInfo info, final Message msg) throws RuleServiceException
    {
		AssertArgument.isNotNull(info, "ruleInfo" );
		final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForFileBasedRules(info.getRuleSource(), info.getDslSource(), info.getReload());
		return ruleBaseState.executeStatefulRules(info, msg);
    }

    /**
	 * Execute rules from a decision table using the Stateful rule engine API
	 *
	 * @param decisionTable -
	 *            String reference to a file which contains a spreadsheet of rules
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading
	 *            the ruleSet.
	 * @param message -
	 *            Message that is updated with the results.
	 * @param globals -
	 *            Map of globals variables that should be set in the working memory
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into
	 *            working memory
	 *
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #executeStatefulRulesFromDecisionTable(RuleInfo, Message)} instead.
	 */
	@Deprecated
    public Message executeStatefulRulesFromDecisionTable(
			final String decisionTable,
			final boolean ruleReload,
			Message message,
			final Map<String,Object> globals,
			final List<Object> objectList) throws RuleServiceException
	{
		AssertArgument.isNotNullAndNotEmpty( decisionTable, "decisionTable" );

		final RuleInfoBuilder builder = new RuleInfoBuilder(decisionTable);
		builder.reload(ruleReload);
		builder.globals(globals);
		builder.defaultFacts(objectList);
		StatefulRuleInfoImpl ruleInfo = new StatefulRuleInfoImpl(builder.build(), false, false);

		final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForDecisionTable(ruleInfo.getRuleSource(), ruleInfo.getReload());

		return ruleBaseState.executeStatefulRules(ruleInfo, message);
	}

	/**
	 * Execute rules from a decision table using the Stateful rule engine API
	 *
	 * @param info {@link StatefulRuleInfo} containing the execution information for the Rule engine.
	 * @param msg The ESB Message object.
	 * @return {@link Message} The ESB Message Object.
	 *
	 * @throws RuleServiceException If a problem occurs while trying to create the Drools RuleBase or if an exception
	 * @deprecated Please use {@link #executeStatefulRulesFromRuleAgent(RuleInfo, Message)} instead.
	 */
	@Deprecated
    public Message executeStatefulRulesFromDecisionTable(StatefulRuleInfo ruleInfo, Message msg) throws RuleServiceException
    {
		final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForDecisionTable(ruleInfo.getRuleSource(), ruleInfo.getReload());
		return ruleBaseState.executeStatefulRules(ruleInfo, msg);
    }

    /**
	 * Execute rules using a rule package retrieved via the Rule Agent from a URL of local file system
	 *            using the Stateful rule engine API
	 *
	 * @param ruleAgentProperties -
	 *            String reference to a file which contains properties used by the RuleAgent to locate a rule package.
	 * @param message -
	 *            Message that is updated with the results.
	 * @param globals -
	 *            Map of globals variables that should be set in the working memory
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into
	 *            working memory
	 *
	 * @return Message with updated objects.
	 */
	public Message executeStatefulRulesFromRuleAgent(
			final String ruleAgentProperties,
			Message message,
			final Map<String,Object> globals,
			final List<Object> objectList) throws RuleServiceException
	{
		AssertArgument.isNotNullAndNotEmpty( ruleAgentProperties, "ruleAgentProperties" );

		try
		{
    		final RuleInfoBuilder builder = new RuleInfoBuilder(ruleAgentProperties);
    		builder.globals(globals);
    		builder.defaultFacts(objectList);
    		StatefulRuleInfoImpl ruleInfo = new StatefulRuleInfoImpl(builder.build(), false, false);

    		final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForRuleAgent(ruleInfo.getRuleSource()) ;

			return ruleBaseState.executeStatefulRules(ruleInfo, message);
		}
		catch (IOException e)
		{
			throw new RuleServiceException( "Could not read the ruleAgentProperties. " + e.getMessage(), e);
		}
		catch (RuleServiceException e)
		{
			throw e;
		}
		catch (Exception e)
		{
			throw new RuleServiceException( "RuleAgent could not get the RuleBase. " + e.getMessage(), e);
		}
	}

	public Message executeStatefulRulesFromRuleAgent(StatefulRuleInfo ruleInfo, Message msg) throws RuleServiceException
    {
	    try
        {
            final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForRuleAgent(ruleInfo.getRuleSource()) ;

            return ruleBaseState.executeStatefulRules(ruleInfo, msg);
        }
        catch (IOException e)
        {
            throw new RuleServiceException( "Could not read the ruleAgentProperties. " + e.getMessage(), e);
        }
        catch (RuleServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new RuleServiceException( "RuleAgent could not get the RuleBase. " + e.getMessage(), e);
        }
    }

    /**
	 * Continue executing rules where the working memory already exists.
	 *
	 * @param rules -
	 *            String reference to a file, either the drl file, the decision table,
	 *            or the ruleAgentProperties. This is used to find the ruleBase.
	 * @param dispose -
	 *            if set to true, working memory will be disposed after the rules are fired
	 * @param message -
	 *            Message that is updated with the results.
	 * @param globals -
	 *            Map of globals variables that should be set in the working memory
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into
	 *            working memory
	 *
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #continueStatefulRulesExecution(StatefulRuleInfo, Message)} instead.
	 */
	@Deprecated
    public Message continueStatefulRulesExecution(
			final String rules,
			final boolean dispose,
			Message message,
			final Map<String,Object> globals,
			final List<Object> objectList) throws RuleServiceException
	{
		AssertArgument.isNotNullAndNotEmpty( rules, "rules" );

		try
		{
			final Map<String, DroolsRuleBaseState> ruleBaseStates = getCachedRuleBaseStates();
    		final RuleInfoBuilder builder = new RuleInfoBuilder(rules);
    		builder.globals(globals);
    		builder.defaultFacts(objectList);
    		StatefulRuleInfo ruleInfo = new StatefulRuleInfoImpl(builder.build(), dispose, true);

			final DroolsRuleBaseState ruleBaseState = ruleBaseStates.get( rules );

			return ruleBaseState.executeStatefulRules(ruleInfo, message);
		}
		catch (Exception e)
		{
			throw new RuleServiceException( "Could not continue rule execution. " + e.getMessage(), e);
		}
	}

    public Message continueStatefulRulesExecution(StatefulRuleInfo info, Message msg) throws RuleServiceException
    {
		try
		{
			final Map<String, DroolsRuleBaseState> ruleBaseStates = getCachedRuleBaseStates();
			final DroolsRuleBaseState ruleBaseState = ruleBaseStates.get(info.getRuleSource());

			return ruleBaseState.executeStatefulRules(info, msg);
		}
		catch (Exception e)
		{
			throw new RuleServiceException( "Could not continue rule execution. " + e.getMessage(), e);
		}
    }

	public void setConfigTree( final ConfigTree configTree )
	{
	}

	//	package protected methods

	/**
	 * Determine if file based rules need reloading and return the rulebase state
	 *
	 * @param ruleSet -
	 *            String reference to a file which contains a ruleSet.
	 * @param dsl -
	 *            String reference to a file which contains a custom rule language
	 *            definition
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading the
	 *            ruleSet.
	 *
	 * @return Message with updated objects.
	 */
	DroolsRuleBaseState getRuleBaseStateForFileBasedRules(
			final String ruleSet,
			final String dsl,
			final boolean ruleReload) throws RuleServiceException
	{
		try
		{
			final DroolsRuleBaseHelper rbHelper = DroolsRuleBaseHelper.getInstance();


			final ConcurrentHashMap<String, String> ruleSets = lifecycleRuleSets.getLifecycleResource();

			synchronized(ruleSets)
			{
				String newRuleSet = null;
				if ( ruleReload )
				{
					newRuleSet = rbHelper.getRulesAsString( ruleSet, dsl );
					String oldRuleSet = ruleSets.put( ruleSet, newRuleSet );
					if ((oldRuleSet != null) && !oldRuleSet.equals(newRuleSet) )
					{
						final DroolsRuleBaseState ruleBaseState = getCachedRuleBaseStates().remove(ruleSet);
						if (ruleBaseState != null)
						{
							ruleBaseState.dispose() ;
						}
					}
				}

				
				final Map<String, DroolsRuleBaseState> ruleBaseStates = getCachedRuleBaseStates();
				DroolsRuleBaseState ruleBaseState = ruleBaseStates.get(ruleSet) ;
				if (ruleBaseState == null)
				{
					final RuleBase ruleBase = rbHelper.createRuleBaseFromRuleFiles(ruleSet, dsl);
					if (newRuleSet == null)
						newRuleSet = rbHelper.getRulesAsString(ruleSet, dsl);
					ruleSets.put(ruleSet, newRuleSet);
					ruleBaseState = new DroolsRuleBaseState(ruleBase) ;
					ruleBaseStates.put(ruleSet, ruleBaseState) ;
				}

				return ruleBaseState;
			}
		}
		catch (final LifecycleResourceException e)
		{
			throw new RuleServiceException("Could not load lifecycle data. " + e.getMessage(), e);
		}
		catch (final IOException e)
		{
			throw new RuleServiceException("Could not read the rules. " + e.getMessage(), e);
		}
		catch (final DroolsParserException e)
		{
			throw new RuleServiceException("Could not parse the rules. " + e.getMessage(), e);
		}
	}

	/**
	 * Determine if decision table need reloading and return the rulebase state
	 *
	 * @param decisionTable -
	 *            String reference to a file which contains a decision table.
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading the
	 *            ruleSet.
	 *
	 * @return Message with updated objects.
	 */
	DroolsRuleBaseState getRuleBaseStateForDecisionTable(
			final String decisionTable,
			final boolean ruleReload ) throws RuleServiceException {

		try
		{
            Map<String, DroolsRuleBaseState> ruleBaseStates = getCachedRuleBaseStates();

            DroolsRuleBaseState ruleBaseState = (!ruleReload ? ruleBaseStates.get( decisionTable ) : null) ;
            if (ruleBaseState == null)
            {
                synchronized (ruleBaseStates)
                {
                    if (!ruleReload)
                    {
                        ruleBaseState = ruleBaseStates.get( decisionTable );
                    }
                    
                    if (ruleBaseState == null)
                    {
                        final RuleBase ruleBase = DroolsRuleBaseHelper.getInstance().createRuleBaseFromDecisionTable(decisionTable);
                        ruleBaseState = new DroolsRuleBaseState(ruleBase) ;
                        final DroolsRuleBaseState origRuleBasedState = ruleBaseStates.put( decisionTable, ruleBaseState );
                        if (origRuleBasedState != null)
                        {
                            origRuleBasedState.dispose() ;
                        }
                    }
                }
            }
            return ruleBaseState;
        }
		catch (final IOException e)
		{
			throw new RuleServiceException("Could not read the rules from [" + decisionTable + "]", e);
		}
		catch (final DroolsParserException e)
		{
			throw new RuleServiceException("Could not parse the rules in [" + decisionTable + "]", e);
		}
		catch (final LifecycleResourceException e)
		{
			throw new RuleServiceException("Caught a LifecycleResourceException :", e);
		}
	}

	DroolsRuleBaseState getRuleBaseStateForRuleAgent( final String ruleAgentProperties ) throws IOException, Exception
	{
		Map<String, RuleAgent> ruleAgents = getCachedRuleAgents();
		RuleAgent ruleAgent = ruleAgents.get( ruleAgentProperties );
		if ( ruleAgent == null )
		{
			synchronized (ruleAgents)
			{
				ruleAgent = ruleAgents.get( ruleAgentProperties );
				if (ruleAgent == null)
				{
					ruleAgent = DroolsRuleBaseHelper.getInstance().createRuleAgent( ruleAgentProperties );
					ruleAgents.put( ruleAgentProperties, ruleAgent );
				}
			}
		}

		RuleBase currentRuleBase = ruleAgent.getRuleBase();
		//	always update the cache as the rulebase might have been updated.
		final Map<String, DroolsRuleBaseState> ruleBaseStates = getCachedRuleBaseStates() ;
		DroolsRuleBaseState ruleBaseState = ruleBaseStates.get(ruleAgentProperties) ;
		if ((ruleBaseState == null) || (ruleBaseState.getRuleBase() != currentRuleBase))
		{
			synchronized(ruleBaseStates)
			{
				ruleBaseState = ruleBaseStates.get(ruleAgentProperties) ;
				currentRuleBase = ruleAgent.getRuleBase();
				if ((ruleBaseState == null) || (ruleBaseState.getRuleBase() != currentRuleBase))
				{
					if (ruleBaseState != null)
					{
						ruleBaseState.dispose() ;
					}
					ruleBaseState = new DroolsRuleBaseState(currentRuleBase) ;
					ruleBaseStates.put(ruleAgentProperties, ruleBaseState) ;
				}
			}
		}
		return ruleBaseState;
	}

	Map<String, RuleAgent> getCachedRuleAgents() throws LifecycleResourceException
	{
		return lifecycleRuleAgents.getLifecycleResource();
	}

	Map<String, DroolsRuleBaseState> getCachedRuleBaseStates() throws LifecycleResourceException
	{
		return lifecycleRuleBaseStates.getLifecycleResource();
	}

	/**
	 * The lifecycle resource factory for rule base state.
	 *
	 * @author kevin
	 */
	public static class LifecycleRuleBaseStateFactory implements LifecycleResourceFactory<Map<String, DroolsRuleBaseState>>
	{
		/**
		 * Create a resource object which will be associated with the
		 * specified lifecycle identity.
		 *
		 * @param lifecycleIdentity
		 *            The associated lifecycle identity.
		 * @return The lifecycle resource
		 * @throws LifecycleResourceException
		 *             for errors during construction.
		 */
		public Map<String, DroolsRuleBaseState> createLifecycleResource( final String lifecycleIdentity) throws LifecycleResourceException
		{
			return new ConcurrentHashMap<String, DroolsRuleBaseState>();
		}

		/**
		 * Destroy a resource object which is associated with the specified lifecycle identity.
		 *
		 * @param resource
		 *            The lifecycle resource.
		 * @param lifecycleIdentity
		 *            The associated lifecycle identity.
		 * @return The lifecycle resource.
		 * @throws LifecycleResourceException
		 *             for errors during destroy.
		 */
		public void destroyLifecycleResource( final Map<String, DroolsRuleBaseState> resource, final String lifecycleIdentity) throws LifecycleResourceException
		{
		    if (resource != null)
		    {
		        final Collection<DroolsRuleBaseState> values = resource.values();
		        for (DroolsRuleBaseState ruleBaseState : values)
		        {
		            ruleBaseState.dispose() ;
		        }
		        resource.clear();
		    }
		}
	}

	public static class LifecycleRuleAgentFactory implements LifecycleResourceFactory<Map<String, RuleAgent>>
	{
        /**
         * Create a resource object which will be associated with the
         * specified lifecycle identity.
         *
         * @param lifecycleIdentity
         *            The associated lifecycle identity.
         * @return The lifecycle resource
         * @throws LifecycleResourceException
         *             for errors during construction.
         */
        public Map<String, RuleAgent> createLifecycleResource( final String lifecycleIdentity) throws LifecycleResourceException
        {
        	return new ConcurrentHashMap<String, RuleAgent>();
        }

        /**
         * Destroy a resource object which is associated with the specified
         * lifecycle identity.
         *
         * @param resource
         *            The lifecycle resource.
         * @param lifecycleIdentity
         *            The associated lifecycle identity.
         * @return The lifecycle resource.
         * @throws LifecycleResourceException
         *             for errors during destroy.
         */
        public void destroyLifecycleResource( final Map<String, RuleAgent> resource, final String lifecycleIdentity) throws LifecycleResourceException
        {
            if (resource != null)
            {
                final Collection<RuleAgent> values = resource.values();
                for (RuleAgent ruleAgent : values)
                {
                    if (log.isInfoEnabled())
                    {
                        log.info("destroying rule agent for " + lifecycleIdentity);
                    }
                    ruleAgent.stopPolling() ;
                    final StatefulSession[] sessions = ruleAgent.getRuleBase().getStatefulSessions();
                    if (sessions != null)
                    {
                        for (StatefulSession statefulSession : sessions)
                        {
                            statefulSession.dispose();
                        }
                    }
                }
                resource.clear();
            }
        }
    }

	/**
	 * The lifecycle resource factory for rule sets.
	 *
	 * @author kevin
	 */
	public static class LifecycleRuleSetFactory implements LifecycleResourceFactory<ConcurrentHashMap<String, String>>
	{
		/**
		 * Create a resource object which will be associated with the
		 * specified lifecycle identity.
		 *
		 * @param lifecycleIdentity
		 *            The associated lifecycle identity.
		 * @return The lifecycle resource
		 * @throws LifecycleResourceException
		 *             for errors during construction.
		 */
		public ConcurrentHashMap<String, String> createLifecycleResource( final String lifecycleIdentity) throws LifecycleResourceException
		{
			return new ConcurrentHashMap<String, String>();
		}

		/**
		 * Destroy a resource object which is associated with the specified
		 * lifecycle identity.
		 *
		 * @param resource
		 *            The lifecycle resource.
		 * @param lifecycleIdentity
		 *            The associated lifecycle identity.
		 * @return The lifecycle resource.
		 * @throws LifecycleResourceException
		 *             for errors during destroy.
		 */
		public void destroyLifecycleResource( final ConcurrentHashMap<String, String> resource, final String lifecycleIdentity) throws LifecycleResourceException
		{
		    if (resource != null)
		    {
		        resource.clear();
		    }
		}
	}
}
