/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.services.rules;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.log4j.Logger;
import org.drools.RuleBase;
import org.drools.SessionConfiguration;
import org.drools.StatefulSession;
import org.drools.StatelessSession;
import org.drools.WorkingMemoryEntryPoint;
import org.drools.common.EventFactHandle;
import org.drools.common.InternalFactHandle;
import org.drools.impl.EnvironmentFactory;
import org.drools.runtime.Environment;
import org.drools.spi.GlobalResolver;
import org.jboss.internal.soa.esb.services.rules.util.RulesContext;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.rules.StatefulRuleInfo;

/**
 * Drools rule base state used to execute and cleanup rule bases.  Some parts have been extracted from the DroolsRuleService.
 * 
 * @author <a href='mailto:Kevin.Conner@jboss.com'>Kevin Conner</a>
 * @author jdelong@redhat.com
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 */
class DroolsRuleBaseState
{
    /**
     * The logger for this state.
     */
    private static Logger LOGGER = Logger.getLogger(DroolsRuleBaseState.class);
    
    /**
     * The rule base associated with this state.
     */
    private final RuleBase ruleBase ;
    /**
     * The disposed flag.  If set, stateful sessions are automatically disposed.
     */
    private transient boolean disposed ;
    
    /**
     * The stateful session.
     */
    private StatefulSession statefulSession ;
    /**
     * Stateful session configuration.
     */
    private SessionConfiguration statefulSessionConfiguration ;
    /**
     * Stateful environment
     */
    private Environment statefulEnvironment ;
    /**
     * The stateful session lock.
     */
    private final Lock statefulSessionLock = new ReentrantLock() ;
    /**
     * The stateful session count lock.
     */
    private final Lock statefulSessionCountLock = new ReentrantLock() ;
    /**
     * The number of sessions queued for the stateful session.
     */
    private int statefulSessionCount ;
    /**
     * The stateless sessions.
     */
    private final ConcurrentLinkedQueue<StatelessSession> statelessSessions = new ConcurrentLinkedQueue<StatelessSession>() ;
    
    /**
     * Construct the rule base state.
     * @param ruleBase The associated rule base.
     */
    DroolsRuleBaseState(final RuleBase ruleBase)
    {
        this.ruleBase = ruleBase ;
    }
    
    /**
     * Get the rule base associated with this state.
     * @return The rule base.
     */
    RuleBase getRuleBase()
    {
        return ruleBase ;
    }
    
    /**
     * Execute rules using using the Stateless API
     *
     * @param message -
     *          Message that is updated with the results.
     * @param objectList -
     *          a list with additional objects (typically pulled from the message)
     *          to be inserted into working memory
     * @param globals -
     *            Map of globals variables that should be set in the working memory
     *
     * @return Message -
     *          with updated objects.
     */
    Message executeStatelessRules(
            final Message message,
            final Map<String,Object> globals,
            final List<Object> objectList)
    {
        final StatelessSession head = statelessSessions.poll() ;
        final StatelessSession statelessSession = (head != null ? head : ruleBase.newStatelessSession()) ;
        RulesContext.clearContext() ;
        try
        {
            final List<Object> facts = new ArrayList<Object>() ;
            facts.add(message) ;
            if (objectList != null)
            {
                facts.addAll(objectList) ;
            }
            statelessSession.setGlobalResolver(new StatelessGlobalResolver(globals)) ;
            try
            {
                statelessSession.execute(facts);
            }
            finally
            {
                statelessSession.setGlobalResolver(null) ;
            }
        }
        finally
        {
            RulesContext.clearContext() ;
            statelessSessions.add(statelessSession) ;
        }
        return message;
    }
    
    /**
     * Execute rules using using the Stateful API
     *
     * @param dispose -
     *          if true the working memory will be dereferenced.
     * @param message -
     *          Message that is updated with the results.
     * @param globals -
     *          Map of globals variables that should be set in the working memory
     * @param entryPointMap -
     *          a map with additional objects to be inserted into working memory or
     *          into the named WorkingMemoryEntryPoint. Any objects included in the
     *          {@link #DEFAULT_ENTRY_POINT} name will be inserted into the normal
     *          working memory. The rest will lookup the entry point working memory
     *          and insert the objects into the those working memories.
     * @return Message -
     *          a possibly updated Message object. The message object is available
     *          to Rules and my be updated by them.
     * @throws RuleServiceException
     */
    Message executeStatefulRules(
            final StatefulRuleInfo ruleInfo,
            final Message message) throws RuleServiceException
    {
        final boolean debugEnabled = LOGGER.isDebugEnabled() ;
        
        RulesContext.clearContext() ;
        statefulSessionCountLock.lock() ;
        statefulSessionCount++ ;
        statefulSessionCountLock.unlock() ;
        
        try
        {
            statefulSessionLock.lock() ;
            try
            {
                if ((statefulSession != null) && !ruleInfo.continueState())
                {
                    if (debugEnabled)
                    {
                        LOGGER.debug("Disposing current stateful session, no continue set") ;
                    }
                    final StatefulSession disposedStatefulSession = statefulSession ;
                    statefulSession = null ;
                    disposedStatefulSession.dispose() ;
                }
                
                if (statefulSession == null)
                {
                    if (debugEnabled)
                    {
                        LOGGER.debug("Creating stateful session") ;
                    }
                    if (statefulSessionConfiguration == null)
                    {
                        statefulSessionConfiguration = new SessionConfiguration() ;
                    }
                    if (statefulEnvironment == null)
                    {
                        statefulEnvironment = EnvironmentFactory.newEnvironment() ;
                    }
                    statefulSession = ruleBase.newStatefulSession(statefulSessionConfiguration, statefulEnvironment) ;
                }
                
                try
                {
                    final Map<String, Object> globals = ruleInfo.getGlobals() ;
                    if (globals != null)
                    {
                        if (debugEnabled)
                        {
                            LOGGER.debug("Inserting globals") ;
                        }
                        final Set<Entry<String, Object>> entrySet = globals.entrySet() ;
                        for(Entry<String, Object> entry : entrySet)
                        {
                            statefulSession.setGlobal( entry.getKey(), entry.getValue() );
                        }
                    }

                    // Always insert the ESB Message object.
                    InternalFactHandle handle = (InternalFactHandle) statefulSession.insert(message);
                    if (debugEnabled && handle.isEvent())
                    {
                        EventFactHandle ef = (EventFactHandle) handle;
                        LOGGER.debug("Event :" +  ef.getObject().getClass().getName() + ", startTimeStamp: " + ef.getStartTimestamp());
                    }
                    
                    // Always insert the Default facts
                    final List<Object> defaultFacts = ruleInfo.getDefaultFacts();
                    if (defaultFacts != null)
                    {
                        if (debugEnabled)
                        {
                            LOGGER.debug("Inserting default facts") ;
                        }
                        for(Object object : defaultFacts)
                        {
                            statefulSession.insert(object);
                        }
                    }

                    // Insert any object that that did not specify an entry-point into the main working memory.
                    if (ruleInfo.getFacts() != null)
                    {
                        if (debugEnabled)
                        {
                            LOGGER.debug("Inserting facts") ;
                        }
                        for(Entry<String, List<Object>> entry : ruleInfo.getFacts().entrySet())
                        {
                            String entryPointName = entry.getKey();
                            // Insert object that have explicitly specified an entry-point.
                            WorkingMemoryEntryPoint wmep = statefulSession.getWorkingMemoryEntryPoint(entryPointName);
                            if (wmep == null)
                            {
                                throw new RuleServiceException("The entry-point '" + entryPointName + "' was not found in the current stateful session. Please double check your rules source");
                            }
                            for(Object object : entry.getValue())
                            {
                                wmep.insert(object);
                            }
                        }
                    }
                    
                    if (debugEnabled)
                    {
                        LOGGER.debug("Firing rules") ;
                    }
                    // Fire rules.
                    statefulSession.fireAllRules();
                }
                finally
                {
                    if ( ruleInfo.dispose() )
                    {
                        if (debugEnabled)
                        {
                            LOGGER.debug("Disposing current stateful session, dispose set") ;
                        }
                        final StatefulSession disposedStatefulSession = statefulSession ;
                        statefulSession = null ;
                        disposedStatefulSession.dispose() ;
                    }
                }
            }
            finally
            {
                statefulSessionLock.unlock() ;
            }
        }
        finally
        {
            RulesContext.clearContext() ;
            statefulSessionCountLock.lock() ;
            statefulSessionCount-- ;
            if (disposed && (statefulSessionCount == 0))
            {
                dispose() ;
            }
            statefulSessionCountLock.unlock() ;
        }
        return message ;
    }
    
    void dispose()
    {
        statefulSessionCountLock.lock() ;
        try
        {
            disposed = true ;
            if ((statefulSessionCount == 0) && (statefulSession != null))
            {
                LOGGER.debug("Disposing stateful session") ;
                final StatefulSession disposedStatefulSession = statefulSession ;
                statefulSession = null ;
                disposedStatefulSession.dispose() ;
            }
            statelessSessions.clear() ;
        }
        finally
        {
            statefulSessionCountLock.unlock() ;
        }
    }
    
    /**
     * Global resolver for stateless execution
     * @author <a href='kevin.conner@jboss.com'>Kevin Conner</a>
     */
    public class StatelessGlobalResolver implements GlobalResolver
    {
        /**
         * The map of globals.
         */
        private final Map<String, Object> globals ;
        
        public StatelessGlobalResolver(Map<String, Object> globals)
        {
            this.globals = new HashMap<String, Object>(globals) ;
        }

        /**
         * Resolve the global identifier.
         * @param identifier The identifier.
         * @return the value or null.
         */
        public Object resolveGlobal(final String identifier)
        {
            return globals.get(identifier) ;
        }

        /**
         * Set the global identifier.
         * @param identifier The identifier.
         * @parma value the value.
         */
        public void setGlobal(final String identifier, final Object value)
        {
            globals.put(identifier, value) ;
        }
    }
}
