/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.jbpm.actions;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import java.io.IOException;
import java.io.Serializable;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SealedObject;
import javax.crypto.SecretKey;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.security.SecurityContext;
import org.jboss.soa.esb.services.security.SecurityService;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.junit.Before;
import org.junit.Test;

/**
 * Unit test for {@link BpmProcessor}.
 *
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class BpmProcessorUnitTest
{
    private BpmProcessor bpmProcessor;
    private Message message;
    private ConfigTree configTree;

    @Test
    public void verifyThatSecurityContextWasAddedToMesssageContext() throws ConfigurationException, InvalidKeyException, NoSuchAlgorithmException, NoSuchPaddingException, IllegalBlockSizeException, IOException
    {
        final SealedObject securityContext = createSealedObject("dummy string");
        SecurityContext.setSecurityContext(securityContext);
        bpmProcessor.addSecurityContext(message);

        assertNotNull(message.getContext().getContext(SecurityService.CONTEXT));
        assertEquals(securityContext, message.getContext().getContext(SecurityService.CONTEXT));
    }

    @Test
    public void verifyThatNullSecurityContextCanBeAddedToMesssageContext() throws ConfigurationException, InvalidKeyException, NoSuchAlgorithmException, NoSuchPaddingException, IllegalBlockSizeException, IOException
    {
        SecurityContext.setSecurityContext(null);
        bpmProcessor.addSecurityContext(message);

        assertNull(message.getContext().getContext(SecurityService.CONTEXT));
    }

    @Test
    public void verifyThatAuthenticationRequestWasAddedToMessageContext() throws ConfigurationException
    {
        //  Doesn't need to be encrypted as this is only passed along and never descrypted.
        byte[] encryptedAuthRequest = new byte[100];
        AuthenticationRequestImpl.setEncryptedAuthRequest(encryptedAuthRequest);
        bpmProcessor.addAuthenticationRequest(message);

        assertNotNull(message.getContext().getContext(SecurityService.AUTH_REQUEST));
        assertEquals(encryptedAuthRequest, message.getContext().getContext(SecurityService.AUTH_REQUEST));
    }

    @Test
    public void verifyThatNullAuthenticationRequestCanBeAddedToMessageContext() throws ConfigurationException
    {
        AuthenticationRequestImpl.setEncryptedAuthRequest(null);
        bpmProcessor.addAuthenticationRequest(message);
        assertNull(message.getContext().getContext(SecurityService.AUTH_REQUEST));
    }

    @Before
    public void setup() throws ConfigurationException
    {
        configTree = new ConfigTree("junit");
        bpmProcessor = new BpmProcessor(configTree);
        message = MessageFactory.getInstance().getMessage();
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(BpmProcessorUnitTest.class);
    }

    private SealedObject createSealedObject(final Serializable ser) throws NoSuchAlgorithmException, NoSuchPaddingException, IllegalBlockSizeException, IOException, InvalidKeyException
    {
        final KeyGenerator kpg = KeyGenerator.getInstance("TripleDES");
        kpg.init(112);
        final SecretKey secretKey = kpg.generateKey();
        final Cipher cipher = Cipher.getInstance("TripleDES");
        cipher.init(Cipher.ENCRYPT_MODE, secretKey);

        return new SealedObject(ser, cipher);
    }
}
