/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.jbpm.integration.timer;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.jboss.soa.esb.util.ClassUtil;
import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmException;
import org.jbpm.svc.Service;
import org.jbpm.svc.ServiceFactory;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.impl.StdSchedulerFactory;

/**
 * Implementation of a scheduler service using quartz.
 */
public class QuartzSchedulerServiceFactory implements ServiceFactory
{
    /**
     * Serial version uid for this class.
     */
    private static final long serialVersionUID = -854771628720741785L;
    /**
     * The instance name property.
     */
    public static final String INSTANCE_NAME = "org.quartz.scheduler.instanceName" ;
    /**
     * The thread name property.
     */
    public static final String THREAD_NAME = "org.quartz.scheduler.threadName" ;
    /**
     * The thread count property.
     */
    public static final String THREAD_COUNT = "org.quartz.threadPool.threadCount" ;
    /**
     * The default maximum refire count.
     */
    public static final int DEFAULT_MAX_REFIRE_COUNT = 5 ;
    
    /**
     * Quartz configuration.
     */
    String quartzConfiguration = "jbpm-quartz.properties" ;
    /**
     * Quartz maximum refire count.
     */
    int maxRefireCount = DEFAULT_MAX_REFIRE_COUNT ;
    /**
     * The quartz scheduler associated with this factory.
     */
    private Scheduler scheduler ;
    
    /**
     * Construct the quartz scheduler service factory.
     * @throws JbpmException For errors during construction.
     */
    public QuartzSchedulerServiceFactory()
        throws JbpmException
    {
        final InputStream quartzProperties = ClassUtil.getResourceAsStream("/" + quartzConfiguration, getClass()) ;

        if(quartzProperties == null)
        {
            throw new JbpmException("Failed to locate the default scheduling properties") ;
        }

        final Properties properties = new Properties();
        try
        {
            properties.load(quartzProperties) ;
        }
        catch (final IOException ioe)
        {
            throw new JbpmException("Failed to load the default scheduling properties") ;
        }
        
        final String name = "JbpmSchedulerService" ;
        properties.put(INSTANCE_NAME, name) ;
        properties.put(THREAD_NAME, name) ;
        if (!properties.containsKey(THREAD_COUNT))
        {
            properties.put(THREAD_COUNT, "1") ;
        }
        final Scheduler scheduler ;
        try
        {
            scheduler = new StdSchedulerFactory(properties).getScheduler();
            scheduler.start() ;
        }
        catch (final SchedulerException se)
        {
            throw new JbpmException("Failed to initialise the scheduler", se) ;
        }
        this.scheduler = scheduler ;
        if (maxRefireCount < 0)
        {
            maxRefireCount = DEFAULT_MAX_REFIRE_COUNT ;
        }
    }

    /**
     * Create an instance of the scheduler service.
     */
    public Service openService()
    {
        return new QuartzSchedulerService(JbpmConfiguration.getInstance().getCurrentJbpmContext(), scheduler, maxRefireCount) ;
    }

    /**
     * Close the service factory.
     */
    public void close()
    {
        if (scheduler != null)
        {
            try
            {
                if (!scheduler.isShutdown())
                {
                    scheduler.shutdown() ;
                }
            }
            catch (final SchedulerException se)
            {
                throw new JbpmException("Failed to shutdown scheduler", se) ;
            }
            finally
            {
                scheduler = null ;
            }
        }
    }
}
