/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.jbpm.integration.msg;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.jboss.soa.esb.services.jbpm.integration.AbstractJmsServiceFactory;
import org.jbpm.JbpmException;
import org.jbpm.svc.Service;

/**
 * Based on the jBPM JmsMessageServiceFactory, this service delivers messages to the
 * job/command listeners over JCA/JMS inflow.
 * 
 * <h3>Configuration</h3>
 * 
 * The JMS message service factory exposes the following configurable fields.
 * 
 * <ul>
 * <li><code>connectionFactoryJndiName</code></li>
 * <li><code>destinationJndiName</code></li>
 * <li><code>commandJndiName</code></li>
 * <li><code>dlqJndiName</code></li>
 * <li><code>isCommitEnabled</code></li>
 * <li><code>idleInterval</code></li>
 * <li><code>maxIdleInterval</code></li>
 * <li><code>maxRetryJobs</code></li>
 * <li><code>providerAdapterJNDI</code></li>
 * </ul>
 * 
 * @author Tom Baeyens
 * @author Alejandro Guizar
 */
public final class JmsMessageServiceFactory extends AbstractJmsServiceFactory
{
  private static final long serialVersionUID = 1L;

  /**
   * The JNDI name for locating the jBPM job queue destination.
   */
  String destinationJndiName = "queue/JbpmJobQueue";
  /**
   * The JNDI name for locating the jBPM command queue destination.
   */
  String commandJndiName = "queue/JbpmCommandQueue";
  /**
   * The JNDI name for locating the jBPM dlq destination.
   */
  String dlqJndiName = "queue/JbpmDLQ";
  /**
   * The default idle interval for the retry executor.
   */
  int idleInterval = 15000; // 15 secs
  /**
   * The maximum idle interval for the retry executor.
   */
  int maxIdleInterval = 3600000; // 1 hour
  /**
   * The maximum number of jobs to reschedule on each poll.
   */
  int maxRetryJobs = 20;

  /**
   * The JMS destination for the jBPM job queue.
   */
  private Destination destination;
  /**
   * The JMS destination for the jBPM command queue.
   */
  private Destination commandDestination;
  /**
   * The JMS destination for the jBPM dlq.
   */
  private Destination dlqDestination;
  /**
   * The retry executor.
   */
  private RetryExecutor retryExecutor ;
  /**
   * The thread associated with the retry executor.
   */
  private Thread retryExecutorThread ;

  /**
   * Construct the JMS message service factory.
   */
  public JmsMessageServiceFactory()
  {
    retryExecutor = new RetryExecutor(idleInterval, maxIdleInterval, maxRetryJobs) ;
    retryExecutorThread = new Thread(retryExecutor) ;
    retryExecutorThread.start() ;
  }

  /**
   * Get the jBPM job queue destination associated with this message service.
   * @return The associated jBPM job queue.
   */
  public Destination getDestination()
  {
    if (destination == null)
    {
      try
      {
        destination = (Destination)lookup(destinationJndiName);
      }
      catch (NamingException e)
      {
        throw new JbpmException("could not retrieve job destination", e);
      }
    }
    return destination;
  }

  /**
   * Get the jBPM command queue destination associated with this message service.
   * @return The associated jBPM command queue.
   */
  public Destination getCommandDestination()
  {
    if (commandDestination == null)
    {
      try
      {
        commandDestination = (Destination)lookup(commandJndiName);
      }
      catch (NamingException e)
      {
        throw new JbpmException("could not retrieve command destination", e);
      }
    }
    return commandDestination;
  }

  /**
   * Get the jBPM dlq destination associated with this message service.
   * @return The associated jBPM dlq.
   */
  public Destination getDLQDestination()
  {
    if (dlqDestination == null)
    {
      try
      {
          dlqDestination = (Destination)lookup(dlqJndiName);
      }
      catch (NamingException e)
      {
        throw new JbpmException("could not retrieve dlq destination", e);
      }
    }
    return dlqDestination;
  }

  /**
   * Create a JMS message service based on this factory.
   */
  public Service openService()
  {
    try
    {
      return new JmsMessageService(getConnectionFactory(), getDestination(), isCommitEnabled());
    }
    catch (JMSException e)
    {
      throw new JbpmException("couldn't open message session", e);
    }
  }

  /**
   * Close this factory.
   */
  public void close()
  {
    super.close() ;
    destination = null;
    commandDestination = null;
    dlqDestination = null;
    try
    {
        retryExecutor.terminate() ;
        try
        {
            retryExecutorThread.join() ;
        }
        catch (final InterruptedException ie) {}
    }
    finally
    {
        retryExecutorThread = null ;
    }
  }
}
