/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.jbpm.integration;

import java.io.Serializable;

import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageProducer;
import javax.jms.Session;

import org.apache.log4j.Logger;
import org.hibernate.Hibernate;
import org.jbpm.JbpmContext;
import org.jbpm.JbpmException;
import org.jbpm.db.JobSession;
import org.jbpm.job.Job;
import org.jbpm.job.Timer;

/**
 * Abstract JMS service code based on the jBPM implementations.
 * 
 * @author <a href='kevin.conner@jboss.com'>Kevin Conner</a>
 */
public abstract class AbstractJmsJobService implements Serializable {
    /**
     * Serial Version UID for this class.
     */
    private static final long serialVersionUID = 8511949684016059912L;

    /**
     * Name of the JBoss Messaging property which drives scheduled delivery.
     */
    private static final String JMS_JBOSS_SCHEDULED_DELIVERY_PROP_NAME = "JMS_JBOSS_SCHEDULED_DELIVERY" ;
    
    /**
     * The logger for this class.
     */
    protected final Logger log = Logger.getLogger(getClass()) ;
    
    /**
     * The current jBPM context.
     */
    private JbpmContext jbpmContext ;
    /**
     * JMS connection factory.
     */
    private ConnectionFactory connectionFactory ;
    /**
     * JMS connection for delivery.
     */
    private Connection connection ;
    /**
     * JMS session for delivery.
     */
    private Session session ;
    /**
     * JMS destination for delivery.
     */
    private Destination destination ;
    /**
     * JMS message producer for delivery.
     */
    private MessageProducer messageProducer ;
    /**
     * Flag indicating that commit should be invoked.
     */
    private boolean isCommitEnabled ;

    /**
     * Construct the base JMS job service.
     * @param jbpmContext The current jBPM context.
     * @param connectionFactory The associated JMS connection factory.
     * @param destination The JMS destination for jobs.
     * @param isCommitEnabled true if commit should be invoked, false otherwise.
     * @throws JMSException For errors in constructing the JMS session.
     */
    public AbstractJmsJobService(final JbpmContext jbpmContext, ConnectionFactory connectionFactory, Destination destination, boolean isCommitEnabled) {
        this.jbpmContext = jbpmContext ;
        this.connectionFactory = connectionFactory ;

        this.destination = destination;
        this.isCommitEnabled = isCommitEnabled;
      }

    /**
     * Send the timer to the job executor.
     * @param timer The timer to send.
     */
    protected void sendTimer(final Timer timer) {
        send(timer, true) ;
    }
    
    /**
     * Send the job to the job executor.
     * @param job The job to send.
     */
    protected void sendJob(final Job job) {
        send(job, false) ;
    }
    
    /**
     * Send the job to the job executor.
     * @param job The job to send.
     * @param timer true if a timer, false if a job.
     */
    private void send(final Job job, final boolean timer) {
        final String type = timer ? " timer " : " job " ;
        if (log.isDebugEnabled()) {
            log.debug("Sending" + type + job.getId()) ;
        }
        // Force initialisation of any proxy, JBESB-2720
        Hibernate.initialize(job) ;
        getJobSession().saveJob(job);
        try {
            Message message = getSession().createMessage();
            if (job.getToken()!=null) {
                message.setLongProperty("tokenId", job.getToken().getId());
            }
            if (job.getProcessInstance()!=null) {
                message.setLongProperty("processInstanceId", job.getProcessInstance().getId());
            }
            if (job.getTaskInstance()!=null) {
                message.setLongProperty("taskInstanceId", job.getTaskInstance().getId());
            }
            if (timer) {
                message.setLongProperty("timerId", job.getId());
                message.setLongProperty(JMS_JBOSS_SCHEDULED_DELIVERY_PROP_NAME, job.getDueDate().getTime()) ;
            } else {
                message.setLongProperty("jobId", job.getId());
            }
            getMessageProducer().send(message);
            if (log.isDebugEnabled()) {
                log.debug("Sent" + type + job.getId()) ;
            }
        } catch (JMSException e) {
            throw new JbpmException("couldn't send jms message", e);
        }
      }

    /**
     * Close the job service.
     */
    public void close() {
        JbpmException exception = null;

        if (messageProducer!=null) {
            try {
                messageProducer.close();
            } catch (Exception e) {
                // NOTE that Error's are not caught because that might halt the JVM and mask the original Error.
                exception = new JbpmException("couldn't close message producer", e);
            }
        }

        if (session!=null) {
            if (isCommitEnabled) {
                try {
                    session.commit();
                } catch (Exception e) {
                    if (exception==null) {
                        exception = new JbpmException("couldn't commit JMS session", e);
                    }
                }
            }
          
            try {
                session.close();
            } catch (Exception e) {
                if (exception==null) {
                    exception = new JbpmException("couldn't close JMS session", e);
                }
            }
        }

        if (connection!=null) {
            try {
                connection.close();
            } catch (Exception e) {
                if (exception==null) {
                    exception = new JbpmException("couldn't close JMS connection", e);
                }
            }
        }

        if (exception!=null) {
            log.debug("Exception during close", exception) ;
            throw exception;
        }
    }

    /**
     * Get the JMS session associated with this job service.
     * @return The JMS session.
     */
    public Session getSession() throws JMSException {
        if (connection == null) {
            log.debug("Creating connection") ;
            connection = connectionFactory.createConnection() ;
        }
        
        if (session == null) {
            log.debug("Creating session") ;
            /* 
             * If the connection supports XA, the session will always take part in the global transaction.
             * Otherwise the first parameter specifies whether message productions and consumptions 
             * are part of a single transaction (TRUE) or performed immediately (FALSE).
             * Messages are never meant to be received before the database transaction commits,
             * hence the transacted is preferable.
             */
            session = connection.createSession(true, Session.SESSION_TRANSACTED);
        }
        return session;
    }

    /**
     * Get the job session associated with this job service.
     * @return The job session.
     */
    public JobSession getJobSession() {
        return jbpmContext.getJobSession();
    }

    /**
     * Get the associated JMS message producer.
     * @return The JMS message producer.
     * @throws JMSException for errors during construction of the JMS message producer.
     */
    protected MessageProducer getMessageProducer() throws JMSException {
        if (messageProducer==null) {
            messageProducer = getSession().createProducer(destination);
        }
        return messageProducer;
    }
}
