/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.services.registry;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import java.util.Collection;
import java.util.Iterator;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
/**
 * Testing the registry.
 * 
 * @author kstam
 *
 */
public class RegistryUnitTest
{
	private static final String SERVICE_NAME = "Kurts Travel Agency";
	private static final String CATEGORY = "travel";
	
	private static Logger logger = Logger.getLogger(RegistryUnitTest.class);
	
	/**
	 * Testing the registering of an EPR.
	 */
	@Test
	public void publishEPR() throws RegistryException
	{
		EPR epr = new EPR();
		Registry registry = RegistryFactory.getRegistry();
		registry.registerEPR(CATEGORY, SERVICE_NAME, "Service for traveling", 
				epr, "Specific Service Binding for traveling");
		//make sure we can handle some null arguments
		try
		{
			registry.registerEPR(null, SERVICE_NAME, null,
					epr, null);
			fail("Expected exception") ;
		}
		catch (final RegistryException re) {} // expected
	}
	/**
	 * This test will try to find all services in the "travel" category.
	 */
	@Test
	public void findService() throws RegistryException
	{
		Registry registry = RegistryFactory.getRegistry();
		Collection<String> services = registry.findServices(CATEGORY);
		for (Iterator i=services.iterator();i.hasNext();) {
			String serviceName = (String) i.next();
			logger.log(Level.INFO, "Found service: " + serviceName);
			assertEquals(SERVICE_NAME, serviceName);
		}
	}
	/**
	 * Find the EPRs for this service.
	 */
	@Test
	public void findEPRs() throws RegistryException, ServiceNotFoundException
	{
		Registry registry = RegistryFactory.getRegistry();
		Collection<EPR> eprs = registry.findEPRs(CATEGORY, SERVICE_NAME);
		for (Iterator i=eprs.iterator();i.hasNext();) {
			EPR epr = (EPR) i.next();
			logger.log(Level.INFO, "Found epr: " + epr);
		}
	}
	@Test
	public void unregisterEPR() throws RegistryException, ServiceNotFoundException
	{
		Registry registry = RegistryFactory.getRegistry();
		EPR eprToBeRemoved = new EPR();
		registry.unRegisterEPR(CATEGORY, SERVICE_NAME, eprToBeRemoved);
//		Now make sure this EPR is really gone
		Collection<EPR> eprs = registry.findEPRs(CATEGORY, SERVICE_NAME);
		logger.log(Level.INFO, "Number of Bindings left should be 0, and is: " + eprs.size());
		assertEquals(0, eprs.size());
	}
	@Test
	public void unregisterService() throws RegistryException, ServiceNotFoundException 
	{
		Registry registry = RegistryFactory.getRegistry();
		try
		{
			registry.unRegisterService(null, SERVICE_NAME);
			fail("Expected ServiceNotFoundException") ;
		}
		catch (final ServiceNotFoundException snfe) {} // Expected
		registry.unRegisterService(CATEGORY, SERVICE_NAME);
		//Make sure it's really gone
		Collection<String> services = registry.findAllServices();
		logger.log(Level.INFO, "Number of Services left should be 0, and is: " + services.size());
		assertEquals(services.size(),0);
		try
		{
			registry.unRegisterService(CATEGORY, SERVICE_NAME);
			fail("Expected ServiceNotFoundException") ;
		}
		catch (final ServiceNotFoundException snfe) {} // Expected
	}
	/**
	 * Setup the database.
	 * @throws Exception
	 */
	@BeforeClass
	public static void runBeforeAllTests() throws Exception {
		TestEnvironmentUtil.setESBPropertiesFileToUse();
		TestEnvironmentUtil.startJAXRDatabase();
	}
	/**
	 * Shutdown the database
	 * @throws Exception
	 */
	@AfterClass
	public static void runAfterAllTests() throws Exception {
		TestEnvironmentUtil.stopJAXRDatabase();
	}

	public static junit.framework.Test suite() {
		return new JUnit4TestAdapter(RegistryUnitTest.class);
	}
}
