/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.notification;

import static org.junit.Assert.assertEquals;

import java.io.IOException;

import javax.mail.MessagingException;
import javax.mail.internet.AddressException;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.routing.email.Emailer;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.Email;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.junit.Test;

/**
 * NotifyEmail unit tests.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class NotifyEmailUnitTest
{
	@Test
	public void test_NotifyEmail() throws Exception {
		ConfigTree emailMessageEl = new ConfigTree("email");
		
		emailMessageEl.setAttribute(Email.FROM, "a.b@c.com");
		emailMessageEl.setAttribute(Email.SENDTO, "d.e@f.com");
		emailMessageEl.setAttribute(Email.COPYTO, "g.h@i.com");
		emailMessageEl.setAttribute(Email.MESSAGE, "Hi there!!!");
		
		NotifyEmail ne = new TestNotifyEmail(emailMessageEl, "Hi there!!!\nHello");
        Message message = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
        message.getBody().add("Hello".getBytes());
		ne.sendNotification(message);
	}

	@Test
    public void test_NotifyEmailNoMessage() throws Exception {
        ConfigTree emailMessageEl = new ConfigTree("email");
                
        emailMessageEl.setAttribute(Email.FROM, "a.b@c.com");
        emailMessageEl.setAttribute(Email.SENDTO, "d.e@f.com");
        emailMessageEl.setAttribute(Email.COPYTO, "g.h@i.com");
                
        NotifyEmail ne = new TestNotifyEmail(emailMessageEl, "Hello");
        Message message = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
        message.getBody().add("Hello".getBytes());
        ne.sendNotification(message);
    }
	
	@Test
    public void overrideHost() throws Exception {
	    assertOverride(Email.HOST, "b2.host2");
    }
	
	@Test
    public void overridePort() throws Exception {
	    assertOverride(Email.PORT, "8021");
    }
	
	@Test
    public void overrideUsername() throws Exception {
	    assertOverride(Email.USERNAME, "daniel");
    }
	
	@Test
    public void overridePassword() throws Exception {
	    assertOverride(Email.PASSWORD, "letmein");
    }
	
	@Test
    public void overrideAuth() throws Exception {
	    assertOverride(Email.AUTH, "true");
    }
	
	@Test
    public void overrideFrom() throws Exception {
	    assertOverride(Email.FROM, "dbevenius@jboss.com");
    }
    
	@Test
    public void overrideSendTo() throws Exception {
	    assertOverride(Email.SENDTO, "dbevenius@jboss.com");
    }
	
	@Test
    public void overrideCopyTo() throws Exception {
	    assertOverride(Email.COPYTO, "dbevenius@jboss.com");
    }
	
	@Test
    public void overrideSubject() throws Exception {
	    assertOverride(Email.SUBJECT, "Overridden Subject");
    }
	
    public void overrideAttachmentName() throws Exception {
	    assertOverride("msgAttachmentName", "newFile.txt");
    }
	
	private void assertOverride(final String overrideName, final String overrideValue) throws Exception
	{
        ConfigTree emailMessageEl = new ConfigTree("email");
        emailMessageEl.setAttribute(Email.SENDTO, "d.e@f.com");
        emailMessageEl.setAttribute(Email.SUBJECT, "Original Subject");
        emailMessageEl.setAttribute(Email.MESSAGE, "Hi there!!!");
        
        TestNotifyEmail ne = new TestNotifyEmail(emailMessageEl, "Hi there!!!\nHello");
        Message message = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
        
        message.getProperties().setProperty(overrideName, overrideValue);
        message.getBody().add("Hello".getBytes());
        ne.sendNotification(message);
        
        assertEquals(overrideValue, MockEmailer.config.getAttribute(overrideName));
	    
	}
	
	private class TestNotifyEmail extends NotifyEmail 
	{
        private final String message ;
        
		public TestNotifyEmail(ConfigTree config, final String message) throws Exception 
		{
			super(config, new MockEmailer(config, message));
			this.message = message ;
		}
		
		@Override
		protected void sendEmailNotification(ConfigTree messageParams, byte[] notusedintest) throws AddressException, MessagingException 
		{
			assertEquals(message, messageParams.getAttribute(Email.MESSAGE));
		}
	}
	
	private static class MockEmailer extends Emailer
	{
	    private static ConfigTree config;
	    private String message;
	    
        public MockEmailer(ConfigTree config, final String message) throws ConfigurationException
        {
            super(config);
            this.message = message;
        }
        
        @Override
        protected void sendEmail(ConfigTree messageParams, byte[] msgPayload) throws AddressException, MessagingException, IOException
        {
            config = messageParams;
			assertEquals(message, messageParams.getAttribute(Email.MESSAGE));
        }
	}
	
	public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(NotifyEmailUnitTest.class);
    }
	
}
