/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.message.tests;

import java.net.URI;

import junit.framework.TestCase;

import org.jboss.soa.esb.addressing.eprs.HTTPEpr;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;

/**
 * Unit tests for the Message class.
 * 
 * @author Mark Little
 */

public class MessageUnitTest extends TestCase
{
	
	public void testDefaultMessageFields ()
	{
		Message msg = MessageFactory.getInstance().getMessage();
		
		assertEquals((msg.getBody() != null), true);
		assertEquals((msg.getHeader() != null), true);
		assertEquals((msg.getContext() != null), true);
		assertEquals((msg.getAttachment() != null), true);
		assertEquals((msg.getFault() != null), true);
		assertEquals((msg.getProperties() != null), true);
	}
	
	public void testXMLMessageFields ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
		
		assertEquals((msg.getBody() != null), true);
		assertEquals((msg.getHeader() != null), true);
		assertEquals((msg.getContext() != null), true);
		assertEquals((msg.getAttachment() != null), true);
		assertEquals((msg.getFault() != null), true);
		assertEquals((msg.getProperties() != null), true);
	}
	
	public void testSerializedMessageFields ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
		
		assertEquals((msg.getBody() != null), true);
		assertEquals((msg.getHeader() != null), true);
		assertEquals((msg.getContext() != null), true);
		assertEquals((msg.getAttachment() != null), true);
		assertEquals((msg.getFault() != null), true);
		assertEquals((msg.getProperties() != null), true);
	}
	
	public void testSerializedMessageToString () throws Exception
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
		
		msg.getHeader().getCall().setAction(new URI("urn:foo"));
		msg.getHeader().getCall().setFrom(new HTTPEpr("http://foo.bar"));
		
		msg.getBody().add("foobar");
		msg.getBody().add("qwerty", "uiop");
		msg.getBody().add(BytesBody.BYTES_LOCATION, "hello world".getBytes());
		
		msg.getFault().setCode(new URI("urn:because"));
		msg.getFault().setReason("something bad happened");
		
		msg.getAttachment().addItem("should be unnamed");
		msg.getAttachment().put("name", "property");
		
		msg.getProperties().setProperty("name", "value");
		msg.getProperties().setProperty("another name", "another value");
		
		System.out.println("Message is: "+msg.toString());
	}
	
	public void testXMLMessageToString () throws Exception
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
		
		msg.getHeader().getCall().setAction(new URI("urn:foo"));
		msg.getHeader().getCall().setFrom(new HTTPEpr("http://foo.bar"));
		
		msg.getBody().add("foobar");
		msg.getBody().add("qwerty", "uiop");
		msg.getBody().add(BytesBody.BYTES_LOCATION, "hello world".getBytes());
		
		msg.getFault().setCode(new URI("urn:because"));
		msg.getFault().setReason("something bad happened");
		
		msg.getAttachment().addItem("should be unnamed");
		msg.getAttachment().put("name", "property");
		
		msg.getProperties().setProperty("name", "value");
		msg.getProperties().setProperty("another name", "another value");
		
		System.out.println("Message is: "+msg.toString());
	}
        
        public void testSerializedMessageCopy () throws Exception
        {
            Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
                
                msg.getHeader().getCall().setAction(new URI("urn:foo"));
                msg.getHeader().getCall().setFrom(new HTTPEpr("http://foo.bar"));
                
                msg.getBody().add("foobar");
                msg.getBody().add("qwerty", "uiop");
                msg.getBody().add(BytesBody.BYTES_LOCATION, "hello world".getBytes());
                
                msg.getFault().setCode(new URI("urn:because"));
                msg.getFault().setReason("something bad happened");
                
                msg.getAttachment().addItem("should be unnamed");
                msg.getAttachment().put("name", "property");
                
                msg.getProperties().setProperty("name", "value");
                msg.getProperties().setProperty("another name", "another value");
                
                Message theCopy = msg.copy();
                
                if (theCopy == msg)
                    fail();
                
                if (!theCopy.getHeader().getCall().getAction().equals(msg.getHeader().getCall().getAction()))
                    fail();
                
                if (!theCopy.getBody().get().equals(msg.getBody().get()))
                    fail();
                
                if (!theCopy.getFault().getCode().equals(msg.getFault().getCode()))
                    fail();
                
                if (!theCopy.getAttachment().get("name").equals(msg.getAttachment().get("name")))
                    fail();
                
                if (!theCopy.getProperties().getProperty("name").equals(msg.getProperties().getProperty("name")))
                    fail();
        }
        
        public void testXMLMessageCopy () throws Exception
        {
            Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
                
                msg.getHeader().getCall().setAction(new URI("urn:foo"));
                msg.getHeader().getCall().setFrom(new HTTPEpr("http://foo.bar"));
                
                msg.getBody().add("foobar");
                msg.getBody().add("qwerty", "uiop");
                msg.getBody().add(BytesBody.BYTES_LOCATION, "hello world".getBytes());
                
                msg.getFault().setCode(new URI("urn:because"));
                msg.getFault().setReason("something bad happened");
                
                msg.getAttachment().addItem("should be unnamed");
                msg.getAttachment().put("name", "property");
                
                msg.getProperties().setProperty("name", "value");
                msg.getProperties().setProperty("another name", "another value");
                
                Message theCopy = msg.copy();
                
                if (theCopy == msg)
                    fail();
                
                if (!theCopy.getHeader().getCall().getAction().equals(msg.getHeader().getCall().getAction()))
                    fail();
                
                if (!theCopy.getBody().get().equals(msg.getBody().get()))
                    fail();
                
                if (!theCopy.getFault().getCode().equals(msg.getFault().getCode()))
                    fail();
                
                if (!theCopy.getAttachment().get("name").equals(msg.getAttachment().get("name")))
                    fail();
                
                if (!theCopy.getProperties().getProperty("name").equals(msg.getProperties().getProperty("name")))
                    fail();
        }
	
}
