/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.message.mapping;

import java.io.UnsupportedEncodingException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import junit.framework.TestCase;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.FileEpr;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;


/**
 * @author kstam
 *
 */
public class FromMessageMapperUnitTest extends TestCase {

    public void testBytes() throws ObjectMappingException, UnsupportedEncodingException
    {
        ObjectMapper mapper = new ObjectMapper();
        String byteString = "testing";
        byte[] byteArray = mapper.getBytes(byteString);
        assertEquals(new String(byteArray),byteString);
    }
    public void testHeader() throws URISyntaxException, ObjectMappingException {
        ObjectMapper mapper = new ObjectMapper();
        Message message = MessageFactory.getInstance().getMessage();
        List<String> variableList = new ArrayList<String>();
        
        EPR eprTo = new FileEpr("testTo");
        message.getHeader().getCall().setTo(eprTo);
        
        EPR eprFrom = new FileEpr("testFrom");
        message.getHeader().getCall().setFrom(eprFrom);
        
        variableList.add("header.call.to");
        variableList.add("header.call.from");
        variableList.add("'header'.'call'.to");
        variableList.add("'header'.'\\call'.from");
        List<Object> pojos=mapper.createObjectList(message, variableList);
        
        Object object1 = pojos.get(0);
        assertTrue(object1.equals(eprTo));
        
        Object object2 = pojos.get(1);
        assertTrue(object2.equals(eprFrom));
        
        final Object quotedObject1 = pojos.get(2);
        assertEquals(eprTo, quotedObject1);
        
        final Object quotedObject2 = pojos.get(3);
        assertEquals(eprFrom, quotedObject2);
        
    }
    public void testProperty () throws ObjectMappingException {
        ObjectMapper mapper = new ObjectMapper();
        Message message = MessageFactory.getInstance().getMessage();
        List<String> variableList = new ArrayList<String>();
        
        message.getProperties().setProperty("property1", "some object");
        variableList.add("properties.property1");
        variableList.add("'properties'.'property\\1'");
        
        TestPojo testPojo1 = new TestPojo();
        testPojo1.setCount(1);
        testPojo1.setName("test1");
        TestPojo testPojo2 = new TestPojo();
        testPojo2.setCount(1);
        testPojo2.setName("test2");
        testPojo1.setTestPojo(testPojo2);
        message.getProperties().setProperty("property2", testPojo1);
        variableList.add("properties.property2.testPojo");
        variableList.add("'properties'.'property2'.testPojo");
        
        List<Object> pojos=mapper.createObjectList(message, variableList);
        
        Object pojo1 = pojos.get(0);
        assertTrue("some object".equals(pojo1));
        
        final Object quotedPojo1 = pojos.get(1);
        assertEquals("some object", quotedPojo1) ;
        
        Object pojo2 = pojos.get(2);
        assertTrue(testPojo2.equals(pojo2));
        
        final Object quotedPojo2 = pojos.get(3);
        assertEquals(testPojo2, quotedPojo2);
    }
    
    public void testProperty_flatten() throws ObjectMappingException {
        ObjectMapper mapper = new ObjectMapper();
        Message message = MessageFactory.getInstance().getMessage();
        List<String> variableList = new ArrayList<String>();
        
        Collection<TestPojo> allPojos = new ArrayList<TestPojo>();
        
        TestPojo testPojo1 = new TestPojo();
        testPojo1.setCount(1);
        testPojo1.setName("test1");
        
        TestPojo testPojo2 = new TestPojo();
        testPojo2.setCount(1);
        testPojo2.setName("test2");
        
        allPojos.add(testPojo1);
        allPojos.add(testPojo2);
        
        message.getProperties().setProperty("allThePojos", allPojos);
        variableList.add("properties.allThePojos");
        variableList.add("'properties'.'allThePojos'");
        
        List<Object> pojos=mapper.createObjectList(message, variableList);
        
        Object pojo1 = pojos.get(0);
        assertTrue(testPojo1.equals(pojo1));
        
        Object pojo2 = pojos.get(1);
        assertTrue(testPojo2.equals(pojo2));
        
        final Object quotedPojo1 = pojos.get(2);
        assertEquals(testPojo1, quotedPojo1) ;
        
        final Object quotedPojo2 = pojos.get(3);
        assertEquals(testPojo2, quotedPojo2);
    }
    
    public void testAttachment_hashmap() throws ObjectMappingException {
        ObjectMapper mapper = new ObjectMapper();
        Message message = MessageFactory.getInstance().getMessage();
        List<String> variableList = new ArrayList<String>();
        
        message.getAttachment().put("attachment1", "some object");
        variableList.add("attachment.attachment1");
        variableList.add("'attachment'.'attachment1'");
        
        TestPojo testPojo1 = new TestPojo();
        testPojo1.setCount(1);
        testPojo1.setName("test1");
        TestPojo testPojo2 = new TestPojo();
        testPojo2.setCount(1);
        testPojo2.setName("test2");
        testPojo1.setTestPojo(testPojo2);
        message.getAttachment().put("attachment2", testPojo1);
        variableList.add("attachment.attachment2.testPojo");
        variableList.add("'attachment'.'attachment2'.testPojo");
        
        List<Object> pojos=mapper.createObjectList(message, variableList);
        
        Object pojo1 = pojos.get(0);
        assertTrue("some object".equals(pojo1));
        
        final Object quotedPojo1 = pojos.get(0);
        assertEquals("some object", quotedPojo1);
        
        Object pojo2 = pojos.get(2);
        assertTrue(testPojo2.equals(pojo2));  
        
        final Object quotedPojo2 = pojos.get(3);
        assertEquals(testPojo2, quotedPojo2);  
    }
    
    public void testAttachment_list() throws ObjectMappingException{
        ObjectMapper mapper = new ObjectMapper();
        Message message = MessageFactory.getInstance().getMessage();
        List<String> variableList = new ArrayList<String>();
        
        message.getAttachment().addItem("some object");
        variableList.add("attachment.0");
        variableList.add("'attachment'.'0'");
        
        TestPojo testPojo1 = new TestPojo();
        testPojo1.setCount(1);
        testPojo1.setName("test1");
        TestPojo testPojo2 = new TestPojo();
        testPojo2.setCount(1);
        testPojo2.setName("test2");
        testPojo1.setTestPojo(testPojo2);
        message.getAttachment().addItem(testPojo1);
        variableList.add("attachment.1.testPojo");
        variableList.add("'attachment'.'1'.testPojo");
        
        List<Object> pojos=mapper.createObjectList(message, variableList);
        
        Object pojo1 = pojos.get(0);
        assertTrue("some object".equals(pojo1));
        
        final Object quotedPojo1 = pojos.get(1);
        assertEquals("some object", quotedPojo1);
        
        Object pojo2 = pojos.get(2);
        assertTrue(testPojo2.equals(pojo2));
        
        final Object quotedPojo2 = pojos.get(3);
        assertEquals(testPojo2, quotedPojo2);
    }
    
    public void testBody() throws ObjectMappingException {
        ObjectMapper mapper = new ObjectMapper();
        Message message = MessageFactory.getInstance().getMessage();
        List<String> variableList = new ArrayList<String>();
        
        message.getBody().add("body1", "some object");
        variableList.add("body.body1");
        variableList.add("'body'.'body1'");
        
        TestPojo testPojo1 = new TestPojo();
        testPojo1.setCount(1);
        testPojo1.setName("test1");
        TestPojo testPojo2 = new TestPojo();
        testPojo2.setCount(1);
        testPojo2.setName("test2");
        testPojo1.setTestPojo(testPojo2);
        message.getBody().add("body2", testPojo1);
        variableList.add("body.body2.testPojo");
        variableList.add("'body'.'body2'.testPojo");
        
        TestPojo testPojo3 = new TestPojo();
        //if the location is not specified we assume it is the body
        message.getBody().add("pojo3", testPojo3);
        variableList.add("pojo3");
        variableList.add("'pojo3'");
        
        List<Object> pojos=mapper.createObjectList(message, variableList);
        
        Object pojo1 = pojos.get(0);
        assertTrue("some object".equals(pojo1));
        
        final Object quotedPojo1 = pojos.get(1);
        assertEquals("some object", quotedPojo1);
        
        Object pojo2 = pojos.get(2);
        assertTrue(testPojo2.equals(pojo2));  
        
        final Object quotedPojo2 = pojos.get(3);
        assertEquals(testPojo2, quotedPojo2);  
        
        Object pojo3 = pojos.get(4);
        assertTrue(testPojo3.equals(pojo3));
        
        final Object quotedPojo3 = pojos.get(5);
        assertEquals(testPojo3, quotedPojo3);
    }
 
}
