/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.message;

import junit.framework.TestCase;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.actions.ActionUtils;

import java.io.UnsupportedEncodingException;

/**
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class MessagePayloadProxyUnitTest extends TestCase {

    public void test_non_legacy() throws MessageDeliverException {
        Message message = MessageFactory.getInstance().getMessage();
        ConfigTree config = new ConfigTree("x");
        MessagePayloadProxy proxy;

        proxy = new MessagePayloadProxy(config);
        
        // Test normal flow....
        proxy.setPayload(message, "Hi");
        assertEquals("Hi", message.getBody().get());
        assertEquals("Hi", proxy.getPayload(message));

        // Test overriding the get/set location on the message...
        config.setAttribute(MessagePayloadProxy.GET_PAYLOAD_LOCATION, "message-loc");
        config.setAttribute(MessagePayloadProxy.SET_PAYLOAD_LOCATION, "message-loc");
        proxy = new MessagePayloadProxy(config);
        proxy.setPayload(message, "Hi");
        assertEquals("Hi", message.getBody().get("message-loc"));
        assertEquals("Hi", proxy.getPayload(message));

        // Test for no exception on null set...
        proxy.setPayload(message, null);

        // Test for exception on null get...
        try {
            proxy.getPayload(message);
            fail("Expected MessageDeliverException");
        } catch(MessageDeliverException e) {
            assertEquals("Null data found in message location(s): [message-loc]", e.getMessage());
        }

        // Now change the Null get Handling config so as there's no exception...
        proxy.setNullGetPayloadHandling(MessagePayloadProxy.NullPayloadHandling.NONE);
        proxy.getPayload(message);

        // Now change the Null set Handling config so as there is an exception...
        proxy.setNullSetPayloadHandling(MessagePayloadProxy.NullPayloadHandling.EXCEPTION);
        try {
            proxy.setPayload(message, null);
            fail("Expected MessageDeliverException");
        } catch(MessageDeliverException e) {
            assertEquals("Setting null data in message location(s): [message-loc]", e.getMessage());
        }
    }

    public void test_legacy() throws MessageDeliverException, UnsupportedEncodingException {
        Message message = MessageFactory.getInstance().getMessage();
        ConfigTree config = new ConfigTree("x");
        MessagePayloadProxy proxy;

        MessagePayloadProxy.setUseLegacyPatterns(true);

        String stringObj = "Hi";
        
        // Multiple get and set locations...
        proxy = new MessagePayloadProxy(config,
                                        new String[] {ActionUtils.POST_ACTION_DATA, BytesBody.BYTES_LOCATION},
                                        new String[] {ActionUtils.POST_ACTION_DATA, BytesBody.BYTES_LOCATION});
        proxy.setPayload(message, stringObj);
        assertTrue(message.getBody().get(ActionUtils.POST_ACTION_DATA) == stringObj);
        assertTrue(proxy.getPayload(message) == stringObj);
        assertTrue(message.getBody().get(BytesBody.BYTES_LOCATION) instanceof byte[]);
        assertEquals("text/plain", message.getProperties().getProperty(MessagePayloadProxy.MIME_TYPE));
        assertEquals("Hi", new String((byte[])message.getBody().get(BytesBody.BYTES_LOCATION), "UTF-8"));

        // Check that setting null clears everything...
        proxy.setPayload(message, null);
        assertTrue(message.getBody().get(ActionUtils.POST_ACTION_DATA) == null);
        assertTrue(message.getBody().get(BytesBody.BYTES_LOCATION) == null);

        // BytesBody.BYTES_LOCATION as the "primary" location...
        proxy = new MessagePayloadProxy(config,
                                        new String[] {BytesBody.BYTES_LOCATION},
                                        new String[] {BytesBody.BYTES_LOCATION});
        proxy.setPayload(message, stringObj);
        // So the String should be stored as a byte[] in BytesBody.BYTES_LOCATION,
        // But proxy.getPayload() should decode the bytes into a String because the
        // mime type is set to "text/plain".
        assertTrue(message.getBody().get(BytesBody.BYTES_LOCATION) instanceof byte[]);
        assertEquals("text/plain", message.getProperties().getProperty(MessagePayloadProxy.MIME_TYPE));
        assertTrue(proxy.getPayload(message) instanceof String);
        assertEquals("Hi", proxy.getPayload(message));

        // Now try a different Object type (non String).  The functionality being tested here is highly
        // suspect, but we still need to maintain backward compatibility to it...
        proxy.setPayload(message, 123);
        assertTrue(message.getBody().get(BytesBody.BYTES_LOCATION) instanceof byte[]);
        assertEquals("java/java.lang.Integer", message.getProperties().getProperty(MessagePayloadProxy.MIME_TYPE));
        assertTrue(proxy.getPayload(message) instanceof byte[]);
    }

    public void test_nonbytes_to_byteslocation() {
        Message message = MessageFactory.getInstance().getMessage();

        try {
            message.getBody().add(BytesBody.BYTES_LOCATION, "not-bytes");
        } catch(Throwable t) {
            fail("Body implementation has changed.  No longer possible to set non-bytes to BytesBody.BYTES_LOCATION, which makes sense.  Review implications for MessagePayloadProxy re legacy support.");
        }
    }
}
