/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.gateway.mina;

import static org.jboss.soa.esb.listeners.gateway.mina.UdpGatewayConfig.HANDLER_CLASS_ATTR;
import static org.jboss.soa.esb.listeners.gateway.mina.UdpGatewayConfig.HOST_ATTR;
import static org.jboss.soa.esb.listeners.gateway.mina.UdpGatewayConfig.PORT_ATTR;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;

import junit.framework.JUnit4TestAdapter;

import org.apache.commons.net.echo.EchoUDPClient;
import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.gateway.mina.DefaultMessageHandler;
import org.jboss.soa.esb.listeners.gateway.mina.UdpGatewayListener;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

/**
 * Unit test for {@link UdpGatewayListener}. </p>
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class UdpGatewayListenerUnitTest
{
    private static final String SERVICE_CATEGORY = "TestCategory";
    private static final String SERVICE_NAME = "TestService";
    private static final String HOST = "localhost";
    private static final String handlerName = DefaultMessageHandler.class.getName();
    private static final int PORT = 9688;

    private MockCourier mockCourier;
    
    @Test (expected = ConfigurationException.class)
    public void nullHost() throws ConfigurationException, ManagedLifecycleException
    {
        final ConfigTree config = createConfig(null, PORT, handlerName, SERVICE_CATEGORY, SERVICE_NAME);
        new UdpGatewayListener(config).initialise();
    }
    
    @Test (expected = ConfigurationException.class)
    public void nullHanlderClass() throws ConfigurationException, ManagedLifecycleException
    {
        final ConfigTree config = createConfig(null, PORT, null, SERVICE_CATEGORY, SERVICE_NAME);
        new UdpGatewayListener(config).initialise();
    }
    
    @Test (expected = ConfigurationException.class)
    public void nullServiceCategory() throws ConfigurationException, ManagedLifecycleException
    {
        final ConfigTree config = createConfig(null, PORT, handlerName, null, SERVICE_NAME);
        new UdpGatewayListener(config).initialise();
    }
    
    @Test (expected = ConfigurationException.class)
    public void nullServiceName() throws ConfigurationException, ManagedLifecycleException
    {
        final ConfigTree config = createConfig(null, PORT, handlerName, SERVICE_CATEGORY, null);
        new UdpGatewayListener(config).initialise();
    }

    @Test
    public void async() throws ConfigurationException, ManagedLifecycleException, InterruptedException, URISyntaxException, UnknownHostException, IOException
    {
        final ConfigTree config = createConfig(HOST, PORT, handlerName, SERVICE_CATEGORY, SERVICE_NAME);
        UdpGatewayListener gateway = new UdpGatewayListener(config);
        gateway.initialise();
        
        final String payload = "Some content";
        
        sendUdpString(payload, HOST, PORT);

        Thread.sleep(1000l);
        
        Object object = mockCourier.message.getBody().get();
        assertTrue(object instanceof byte[]);
        byte[] bytes = (byte[]) object;
        assertEquals(payload, new String(bytes));

        gateway.doStop();
    }
    
    @Before
    public void setUp() throws Exception
    {
        MockCourierFactory.install();
        MockRegistry.install();
        mockCourier = new MockCourier(true, true);
        MockRegistry.register(SERVICE_CATEGORY, SERVICE_NAME, new EPR(new URI("jms://localhost:8080/queue1")), mockCourier);
    }

    @After
    public void tearDown() throws Exception
    {
        MockRegistry.uninstall();
        MockCourierFactory.uninstall();
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(UdpGatewayListenerUnitTest.class);
    }

    private ConfigTree createConfig(final String host, final int port, final String handlerName, final String categoryName, final String serviceName)
    {
        final ConfigTree config = new ConfigTree("NettyGatewayListenerUnitTest");
        config.setAttribute(ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG, categoryName);
        config.setAttribute(ListenerTagNames.TARGET_SERVICE_NAME_TAG, serviceName);
        config.setAttribute(HOST_ATTR, host);
        config.setAttribute(PORT_ATTR, String.valueOf(port));
        config.setAttribute(HANDLER_CLASS_ATTR, handlerName);
        return config;
    }

    private void sendUdpString(final String payload, final String host, final int port) throws UnknownHostException, IOException
    {
        final EchoUDPClient client = new EchoUDPClient();
        client.open();
        try
        {
            final byte[] writeBuffer = payload.getBytes();
            client.setSoTimeout(3000);
            client.send(writeBuffer, writeBuffer.length, InetAddress.getByName(host), port);
        } 
        finally
        {
            client.close();
        }
    }

}
