/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.config.mappers;

import junit.framework.TestCase;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URI;
import java.util.List;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlException;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.FtpListenerDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.Listener;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.FtpBusDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.JbossesbDocument.Factory;
import org.jboss.soa.esb.listeners.gateway.ReadOnlyRemoteGatewayListener;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Tests the class FtpListenerMapper 
 * 
 * @author Daniel Bevenius
 *
 */
public class FtpListenerMapperUnitTest extends TestCase
{
	@SuppressWarnings("unused")
	private Logger log = Logger.getLogger(FtpListenerMapperUnitTest.class);
	
	private static final String ESB_CONFIG_READ_ONLY = "jbossesb_config_readonly.xml";
	private static final String ESB_CONFIG_ESCAPING = "jbossesb_config_escaping.xml";
	
	private static final String GATEWAY_CLASS_ATTRIBUTE = "gatewayClass";
		
	@Test
	public void testReadOnlyConfiguration() throws Exception
	{
		Element root = createRootElement();
		XMLBeansModel model = getXmlBeanModel( ESB_CONFIG_READ_ONLY );
		
		List<Listener> gateways = model.getGatewayListeners();
		FtpListenerDocument.FtpListener ftpListener = (FtpListenerDocument.FtpListener) gateways.get(0);
		assertEquals ( true, ftpListener.getIsGateway() );
		
		FtpBusDocument.FtpBus ftpBus = (FtpBusDocument.FtpBus) model.getBus( ftpListener.getBusidref() );
		assertNotNull( "FtpBus element should exist", ftpBus );
		assertEquals ( "read-only attribute should be true", true, ftpBus.getFtpMessageFilter().getReadOnly() );
		
		FtpListenerMapper.map( root , ftpListener, model );
		
		Node ftpMapperListener = root.getFirstChild();
		assertNotNull ("ftpMapperListener should not be null", ftpMapperListener ); 
		
		NamedNodeMap attributes = ftpMapperListener.getAttributes();
		Node gatewayClass = attributes.getNamedItem( GATEWAY_CLASS_ATTRIBUTE );
		assertNotNull( "gatewayClass should have been set", gatewayClass );
		assertEquals( ReadOnlyRemoteGatewayListener.class.getName(), gatewayClass.getNodeValue() );
	}
	
	@Test
	public void testEscaping() throws Exception
	{
		final XMLBeansModel model = getXmlBeanModel( ESB_CONFIG_ESCAPING );
		final GatewayGenerator gatewayGenerator = new GatewayGenerator(model);
		final Document gatewayConfig = gatewayGenerator.generate();
		final Element root = gatewayConfig.getDocumentElement() ;
		final NodeList listenerList = root.getElementsByTagName("listener") ;
		assertEquals("Listener count", 1, listenerList.getLength()) ;
		final Node listener = listenerList.item(0) ;
		final NamedNodeMap attributes = listener.getAttributes() ;
		final Node urlNode = attributes.getNamedItem(ListenerTagNames.URL_TAG) ;
		assertNotNull("URL node", urlNode) ;
		final String urlValue = urlNode.getNodeValue() ;
		assertNotNull("URL value", urlValue) ;
		final URI uri = new URI(urlValue) ;
		assertEquals("host", "localhost", uri.getHost()) ;
		assertEquals("port", 21, uri.getPort()) ;
		assertEquals("path", "/directory/####", uri.getPath()) ;
		assertEquals("userInfo", "user%3A##user:password:##password", uri.getUserInfo()) ;
	}
	
	private XMLBeansModel getXmlBeanModel( String fileName ) throws ConfigurationException, IOException, XmlException
	{
		InputStream inputStream = getClass().getResourceAsStream( fileName );
		final Reader reader = new InputStreamReader(inputStream);
		XMLBeansModel model = new XMLBeansModel(Factory.parse(reader).getJbossesb());
		return model;
	}
	
	private Element createRootElement() throws ConfigurationException
	{
		Document doc = YADOMUtil.createDocument();
		Element root = YADOMUtil.addElement(doc, "jbossesb-gateways");
		root.setAttribute("parameterReloadSecs", "1000" );
		return root;
	}
}
