/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners;

import java.util.LinkedList;
import java.util.Queue;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.common.tests.BaseTest;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;

public class ListenerManagerBaseTest extends BaseTest
{
	/**
	 * The message lock.
	 */
	private static Lock lock = new ReentrantLock() ;
	/**
	 * The not empty condition.
	 */
	private static Condition notEmpty = lock.newCondition() ;
	/**
	 * The queue of messages.
	 */
	private static final Queue<String> MESSAGES = new LinkedList<String>() ;
	
	/**
	 * Clear the current message queue.
	 */
	protected static void clearMessages()
	{
		lock.lock() ;
		try
		{
			MESSAGES.clear() ;
		}
		finally
		{
			lock.unlock() ;
		}
	}
	
	/**
	 * Add the message to the message queue.
	 * @param text The message text.
	 */
	protected static void addMessage(final byte[] text)
	{
		lock.lock() ;
		try
		{
			final boolean notify = (MESSAGES.size() == 0) ;
			final String message = new String(text) ;
			MESSAGES.add(message) ;
			if (notify)
			{
				notEmpty.signal() ;
			}
		}
		finally
		{
			lock.unlock() ;
		}
	}
	
	/**
	 * Get the first message from the queue.
	 * @param timeout The maximum time to wait before returning. 
	 * @return The first message in the queue or null on timeout.
	 */
	protected static String getMessage(final long timeout)
	{
            final long endTime = System.currentTimeMillis() + timeout ;
            lock.lock() ;
            try
            {
                while (MESSAGES.size() == 0)
                {
                    final long delay = endTime - System.currentTimeMillis() ;
                    if (delay > 0)
                    {
                        try
                        {
                            notEmpty.await(delay, TimeUnit.MILLISECONDS) ;
                        }
                        catch (final InterruptedException ie) { _logger.error( "Interrupted exception during await : " + ie.getMessage());} // Ignore the interrupt
                    }
                    else
                    {
                        return null ;
                    }
                }
                final String message = MESSAGES.remove() ;
                return message ;
            }
            finally
            {
                lock.unlock();
            }
	}

	protected void runBeforeAllTests() throws Exception {
        TestEnvironmentUtil.setESBPropertiesFileToUse("product", "../product");
        TestEnvironmentUtil.startJAXRDatabase();
	}

	protected final void runAfterAllTests() throws Exception
	{
		TestEnvironmentUtil.stopJAXRDatabase();
	}

	protected static final String TMP_DIR = System
			.getProperty("java.io.tmpdir");

	protected static Logger _logger = Logger
			.getLogger(ListenerManagerBaseTest.class);

	public static class MockMessageAwareAction
	{
		Random _rand = new Random(System.currentTimeMillis());

		ConfigTree _config;

		public MockMessageAwareAction(ConfigTree config)
		{
			_config = config;
		}

		public Message notifyTest(Message message) throws Exception
		{
			addMessage((byte[]) message.getBody().get());

			return message;
		}
	}
}
