/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.common.tests;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.HashMap;

import junit.framework.TestCase;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;

/**
 * Base test class for esb test classes, uses jboss logging.
 * 
 * @author <a href="mailto:arvinder.singh@indigo-logic.com">Arvinder Singh</a>
 * @version <tt>$Revision:$</tt> $Id:$
 */

/*
 * This should probably use its own property file.
 */

public class BaseTest extends TestCase
{

	/** The Database driver used for unit tests */
	public static String DB_DRIVER = "org.jboss.esb.test.db.driver";

	/** The Database connection url used for unit tests */
	public static String DB_URL = "org.jboss.esb.test.db.url";

	/** The Database user name used for unit tests */
	public static String DB_USERNAME = "org.jboss.esb.test.db.user";

	/** The Database password used for unit tests */
	public static String DB_PASSWORD = "org.jboss.esb.test.db.password";
	
	/** The FTP hostname used for unit tests **/
	public static String FTP_HOSTNAME = "org.jboss.esb.test.ftp.hostname" ;
	
	/** The FTP user used for unit tests **/
	public static String FTP_USER = "org.jboss.esb.test.ftp.user" ;
	
	/** The FTP directory used for unit tests **/
	public static String FTP_DIR = "org.jboss.esb.test.ftp.dir" ;
	
	/** The FTP password for unit tests **/
	public static String FTP_PWD = "org.jboss.esb.test.ftp.pwd";

	/** The Logger. */
	protected Logger log = Logger.getLogger(getClass());

	/** Contains user and or deployment specific paramater */
	public static HashMap<String, String> SYSTEM_PARAMETERS;

	/**
	 * Default class constructor.
	 */
	public BaseTest()
	{
        super("DummyName");
		SYSTEM_PARAMETERS = readSystemParameters();
	}

	/**
	 * Utility constructor.
	 * 
	 * @param name
	 *            The test name.
	 */
	public BaseTest(String name)
	{
		super(name);
		SYSTEM_PARAMETERS = readSystemParameters();
	}

	/**
	 * Reading system parameter that are set by the ant build when invoking the
	 * junit call. The parameter values should be set in the
	 * deployment.properties file. If the particular parameter is not found it
	 * will receive a default value.
	 */
	private HashMap<String, String> readSystemParameters()
	{
		// Reading system parameter
		String dbDriver = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CONNECTION_MODULE)
				.getProperty(DB_DRIVER, "org.hsqldb.jdbcDriver");
		String dbUrl = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CONNECTION_MODULE)
				.getProperty(DB_URL, "jdbc:hsqldb:.");
		String dbUsername = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CONNECTION_MODULE)
				.getProperty(DB_USERNAME, "sa");
		String dbPassword = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CONNECTION_MODULE)
				.getProperty(DB_PASSWORD, "");
		String ftpHostname = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CONNECTION_MODULE)
		.getProperty(FTP_HOSTNAME, "localhost:10021");
		String ftpUser = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CONNECTION_MODULE)
		.getProperty(FTP_USER, "anonymous");
		String ftpDir = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CONNECTION_MODULE)
		.getProperty(FTP_DIR, "/pub/DNS");
		String ftpPwd = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CONNECTION_MODULE)
		.getProperty(FTP_PWD, "");

		// Setting the paremeters in the HashMap
		HashMap<String, String> systemParameters = new HashMap<String, String>();
		systemParameters.put(DB_DRIVER, dbDriver);
		systemParameters.put(DB_URL, dbUrl);
		systemParameters.put(DB_USERNAME, dbUsername);
		systemParameters.put(DB_PASSWORD, dbPassword);
		systemParameters.put(FTP_HOSTNAME, ftpHostname);
		systemParameters.put(FTP_USER, ftpUser);
		systemParameters.put(FTP_DIR, ftpDir);
		systemParameters.put(FTP_PWD, ftpPwd);

		log.debug("DB parameters:");
		log.debug("Driver=" + dbDriver);
		log.debug("url=" + dbUrl);
		log.debug("username=" + dbUsername);
		log.debug("password=" + dbPassword);
		log.debug("FTP parameters:");
		log.debug("Hostname=" + ftpHostname);
		log.debug("User=" + ftpUser);
		log.debug("Dir=" + ftpDir);
		log.debug("Password="+ftpPwd);

		return systemParameters;
	}

	protected String getDbUrl()
	{
		return SYSTEM_PARAMETERS.get(DB_URL);
	}

	protected String getDbUser()
	{
		return SYSTEM_PARAMETERS.get(DB_USERNAME);
	}

	protected String getDbPassword()
	{
		return SYSTEM_PARAMETERS.get(DB_PASSWORD);
	}

	protected String getDbDriver()
	{
		return SYSTEM_PARAMETERS.get(DB_DRIVER);
	}
	
	protected String getFtpHostname()
	{
		return SYSTEM_PARAMETERS.get(FTP_HOSTNAME) ;
	}
	
	protected String getFtpUser()
	{
		return SYSTEM_PARAMETERS.get(FTP_USER) ;
	}
	
	protected String getFtpDir()
	{
		return SYSTEM_PARAMETERS.get(FTP_DIR) ;
	}
	
	protected String getFtpPwd ()
	{
		return SYSTEM_PARAMETERS.get(FTP_PWD);
	}

	public Connection getDbConnection() throws SQLException
	{
		registerJdbcDriver();
		return DriverManager.getConnection(getDbUrl(), getDbUser(),
				getDbPassword());
	}

	public Connection getDbConnection(String username, String password)
			throws SQLException
	{
		registerJdbcDriver();
		return DriverManager.getConnection(getDbUrl(), username, password);
	}

	/**
	 * Set the test ObjStoreConfigFile location in the classpath/filesystem.
	 * 
	 * @param filePath
	 *            Classpath/filesystem location.
	 */
	public void setObjStoreConfigFile(String filePath)
	{
		if (filePath == null)
		{
			ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperties()
					.remove(Environment.OBJECT_STORE_CONFIG_FILE);
		} else
		{
			ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).setProperty(
					Environment.OBJECT_STORE_CONFIG_FILE, filePath);
		}
	}

	private void registerJdbcDriver()
	{
		try
		{
			Class.forName(getDbDriver()).newInstance();
		} catch (Exception e)
		{
			log.error(e);
			fail("JDBC Driver [" + getDbDriver()
					+ "] not available in classpath.");
		}
	}
}
