/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.common.tests.BaseTest;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.notification.NotificationList;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.xml.sax.SAXException;


/**
 * Unit tests for the ActionUtils class.
 *
 * @author <a href="mailto:jplenhart@yahoo.com">jplenhart@yahoo.com</a>
 */

public class ActionUtilsUnitTest extends BaseTest
{
    private Logger logger = Logger.getLogger(this.getClass());


    /**
     * Testing the Utility to get the Task
     * before Object
     */
    public void testGetTaskBeforeObject()
    {
        Message msg1 = MessageFactory.getInstance().getMessage(MessageType.DEFAULT_TYPE);
        String anyText = "A bunch of text";
        msg1.getBody().add(ActionUtils.PRE_ACTION_DATA, anyText.getBytes());

        Object obj = ActionUtils.getTaskBeforeObject(msg1);
        String anyTextAfter = new String((byte[]) obj);
        assertTrue(anyText.equals(anyTextAfter));
    }

    /**
     * Testing the Utility to get the Copy
     * The current Message Body Object
     * to the Previous
     */
    public void testCopyCurrentToPrevious()
    {
        String beforeAction = "before action";
        String currentObject = "current action";

        Message msg1 = MessageFactory.getInstance().getMessage(MessageType.DEFAULT_TYPE);
        msg1.getBody().add(ActionUtils.PRE_ACTION_DATA, beforeAction.getBytes());
        msg1.getBody().add(ActionUtils.POST_ACTION_DATA, currentObject.getBytes());
        ActionUtils.copyCurrentToPrevious(msg1);

        // Both PRE_ACTION_DATA and POST_ACTION_DATA should be equal
        Object objBeforeAction = msg1.getBody().get(ActionUtils.PRE_ACTION_DATA);
        Object objCurrentObject = msg1.getBody().get(ActionUtils.POST_ACTION_DATA);
        assertEquals(objBeforeAction, objCurrentObject);
    }

    /**
     * Convenience Method to Obtain a Stream
     * To a Mock File
     */
    private InputStream getStream(String sName)
    {
        InputStream oRet = getClass().getResourceAsStream(sName);

        if (null == oRet)
        {
            try
            {
                oRet = new FileInputStream(sName);
            }
            catch (IOException e)
            {
                /* OK  Just fall through */
            }
        }

        return oRet;
    }


    /**
     * Convenience Method to Build a ConfigTree from a File
     */
    private ConfigTree getConfigTreeFromInputStream(String fileName)
            throws SAXException, IOException
    {
        String pathToFile = new StringBuilder()
                .append(TestEnvironmentUtil.getUserDir("product"))
                .append("rosetta/tests/src/org/jboss/soa/esb/actions/")
                .append(fileName).toString();

        logger.debug("Determining If the File Exists - " + fileName);

        //Make sure this file exists
        File configFile = new File(pathToFile);

        if (!configFile.exists())
        {
            logger.error("File does not Exist - Failing the Test");
            assertTrue(false);
        }
        else
        {
            logger.debug("Found the File");
        }

        ConfigTree confTree = ConfigTree.fromInputStream(getStream(pathToFile));

        return confTree;
    }


    /**
     * Testing the ActionUtils class to verify that the
     * Correct NotifyList is returned based on a Mock ConfigTree
     */
    public void testGetNotifyList()
    {
        try
        {
            ConfigTree confTree =
                    getConfigTreeFromInputStream("NotificationList_01.xml");
            NotificationList list = ActionUtils.getNotifyList(confTree, "OK");

            // Expecting a NotificationList Returned (If one is not returned we get Null)
            assertNotNull(list);

            //Since it is not Null -- Verify the content
            assertEquals(1, list.getAllChildren().length);
            assertEquals(
                    "NotifyConsole",
                    list.getFirstChild("target").getAttribute("class"));
        }
        catch (IOException _ioex)
        {
            logger.error("IOException: " + _ioex.getMessage());
            assertTrue(false);
        }
        catch (SAXException _saxex)
        {
            logger.error("SAXException: " + _saxex.getMessage());
            assertTrue(false);
        }
        catch (Exception _ex)
        {
            logger.error("Exception: " + _ex.getMessage());
            assertTrue(false);
        }
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(ActionUtilsUnitTest.class);
    }
}
