/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.util.embedded.ftp;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Properties;

import org.apache.ftpserver.ftplet.Authority;
import org.apache.ftpserver.ftplet.FtpException;
import org.apache.ftpserver.listener.io.IOListener;
import org.apache.ftpserver.usermanager.BaseUser;
import org.apache.ftpserver.usermanager.PropertiesUserManager;
import org.apache.ftpserver.usermanager.WritePermission;
import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.embedded.EmbeddableException;

/**
 * This class extends EmbeddedFtpServer and can be used without
 * having a configuration file for the ftp server.
 * 
 * By default this class will create and perform operations on the following directories:
 *	/tmp/ftpserver/home/
 *									input/
 *									upload/
 *									error/
 *
 *
 *	The default host is localhost and the default port is 21.
 *	But this can be changed by calling any of the setters before calling start.
 *	
 *	Usage:
 *	NoConfigFileFtpServer ftpServer = new NoConfigFileFtpServer();
 *	ftpServer.setPort( 2221 ) ;
 *	ftpServer.start();
 * ...
 * ftpServer.stop();
 * 
 * @author Daniel Bevenius
 *
 */
public class NoConfigFileFtpServer extends EmbeddedFtpServer
{
	private Logger log = Logger.getLogger( NoConfigFileFtpServer.class );
	
	/* root dir for ftp server */
	private String rootDir = System.getProperty( "java.io.tmpdir" );
	
	/* name of the server directory. This will be created directly below the rootDir */
	private String serverDirName = "ftpserver";
	private File ftpServerDir;
	
	/* name of the user home dir. This will be created directly below the ftpServerDir */
	private String homeDirName = "home";
	private File ftpHomeDir;
	
	/* File that point to the local input directory.  */
	private String inputDirName = "input";
	private String remoteInputDir = "/input";
	private File localInputDir;
	
	/* File that point to the local upload directory.  */
	private String uploadDirName = "upload";
	private File localUploadDir;
	
	/* File that point to the local error directory.  */
	private String errorDirName = "error";
	private File localErrorDir;
	
	/* this file is created by the FTP server. Will be named rootDir/ftpServerDir/user.properties */
	private String userPropertiesFileName = "user.properties";
	private File ftpUserPropertiesFile;
	
	/* ftp connection settings */
	private String host = "localhost";
	private int port = 21;
	private String userName = "anonymous";
	private String password = "letMeIn";
	
	public void start() throws EmbeddableException
	{
		createDirs();
		addProperties( getDefaultProperties() );
		createUsers();
		super.start();
	}

	protected void createUsers()
	{
		try
		{
			PropertiesUserManager propertiesUserManager = new PropertiesUserManager();
			propertiesUserManager.setPropFile(  ftpUserPropertiesFile );
			BaseUser ftpUser = new BaseUser();
			ftpUser.setName( userName );
			ftpUser.setPassword( password );
			ftpUser.setHomeDirectory( ftpHomeDir.getAbsolutePath() );
			Authority[] auths = { new WritePermission() };
			ftpUser.setAuthorities( auths );
	        ftpUser.setEnabled(  true );
	        propertiesUserManager.configure();
			propertiesUserManager.save( ftpUser );
			
			BaseUser adminUser = new BaseUser();
			adminUser.setName( "admin" );
			adminUser.setHomeDirectory( ftpHomeDir.getAbsolutePath() );
			propertiesUserManager.save( adminUser );
		} 
		catch (FtpException e)
		{
			log.error( e );
		}
	}

	protected Properties getDefaultProperties()
	{
		Properties properties = new Properties();
        properties.setProperty("config.listeners.default.port", Integer.toString( port ) );
        properties.setProperty("config.listeners.default.class", IOListener.class.getName() );
        properties.setProperty("config.connection-manager.anonymous-login-enabled", "true");
        properties.setProperty("config.user-manager.prop-file", ftpUserPropertiesFile.getAbsolutePath() );
		return properties;
	}

	protected void createDirs()
	{
			ftpServerDir = new File ( rootDir + File.separator + serverDirName );
			if ( ftpServerDir.exists() )
				ftpServerDir.delete();
			
			ftpServerDir.mkdirs();
	        ftpUserPropertiesFile = new File ( ftpServerDir.getAbsolutePath() + File.separator + userPropertiesFileName);
			
			ftpHomeDir = new File(  ftpServerDir, homeDirName );
			ftpHomeDir.mkdirs();
			
			localInputDir = new File( ftpHomeDir,  inputDirName );
			localInputDir.mkdir();
			
			localUploadDir = new File( ftpHomeDir,  uploadDirName);
			localUploadDir.mkdir();
			
			localErrorDir = new File( ftpHomeDir,  errorDirName);
			localErrorDir.mkdir();
	}

	public  File getFtpServerDir()
	{
		return ftpServerDir;
	}

	public  void setFtpServerDir( File ftpServerDir )
	{
		this.ftpServerDir = ftpServerDir;
	}

	public  File getFtpHomeDir()
	{
		return ftpHomeDir;
	}

	public void setFtpHomeDir( File ftpHomeDir )
	{
		this.ftpHomeDir = ftpHomeDir;
	}

	public  File getFtpPropertiesFile()
	{
		return ftpUserPropertiesFile;
	}

	public void setFtpPropertiesFile( File ftpPropertiesFile )
	{
		this.ftpUserPropertiesFile = ftpPropertiesFile;
	}

	public int getPort()
	{
		return port;
	}

	public void setPort( int port )
	{
		this.port = port;
	}

	public File getLocalInputDir()
	{
		return localInputDir;
	}

	public void setLocalInputDir( File localInputDir )
	{
		this.localInputDir = localInputDir;
	}

	public File getLocalUploadDir()
	{
		return localUploadDir;
	}

	public void setLocalUploadDir( File localUploadDir )
	{
		this.localUploadDir = localUploadDir;
	}

	public String getHost()
	{
		return host;
	}

	public void setHost( String host )
	{
		this.host = host;
	}

	public String getRootDir()
	{
		return rootDir;
	}

	public void setRootDir( String rootDir )
	{
		this.rootDir = rootDir;
	}

	public String getUserName()
	{
		return userName;
	}

	public void setUserName( String userName )
	{
		this.userName = userName;
	}

	public String getPassword()
	{
		return password;
	}

	public void setPassword( String password )
	{
		this.password = password;
	}

	public URL getURL() throws MalformedURLException
	{
		return new URL ( "ftp://" + userName + ":" + password + "@" + host + ":" + port + "/" + remoteInputDir );
	}

	public String getServerDirName()
	{
		return serverDirName;
	}

	public void setServerDirName( String serverDirName )
	{
		this.serverDirName = serverDirName;
	}

	public String getHomeDirName()
	{
		return homeDirName;
	}

	public void setHomeDirName( String homeDirName )
	{
		this.homeDirName = homeDirName;
	}

	public String getInputDirName()
	{
		return inputDirName;
	}

	public void setInputDirName( String inputDirName )
	{
		this.inputDirName = inputDirName;
	}

	public String getUploadDirName()
	{
		return uploadDirName;
	}

	public void setUploadDirName( String uploadDirName )
	{
		this.uploadDirName = uploadDirName;
	}

	public String getErrorDirName()
	{
		return errorDirName;
	}

	public void setErrorDirName( String errorDirName )
	{
		this.errorDirName = errorDirName;
	}

	public File getLocalErrorDir()
	{
		return localErrorDir;
	}

	public void setLocalErrorDir( File localErrorDir )
	{
		this.localErrorDir = localErrorDir;
	}
		
}
