/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.rosetta.pooling;

import junit.framework.TestCase;

import javax.naming.Context;
import javax.naming.NamingException;
import javax.jms.JMSException;
import javax.jms.Session;
import java.util.HashMap;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;

import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.internal.soa.esb.rosetta.pooling.jms.MockJMSXAConnectionFactory;
import org.jboss.internal.soa.esb.rosetta.pooling.jms.MockJMSConnectionFactory;

/**
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class MaxSessionsPerConnectionUnitTest extends TestCase {

    private Map<String, String> jndiEnv = new HashMap<String, String>();

    public MaxSessionsPerConnectionUnitTest() {
        jndiEnv.put(Context.INITIAL_CONTEXT_FACTORY, MockInitialContextFactory.class.getName());
        jndiEnv.put(JMSEpr.CONNECTION_FACTORY_TAG, "ConnectionFactory");
    }

    protected void tearDown() throws Exception {
        TransactionStrategy.setTransactionStrategy(new TransactionStrategy.NullTransactionStrategy());
    }

    public void test_nonXA() throws NamingException, JMSException, ConnectionException {
        int MAX_SESSIONS_PER_CONN = 2;

        jndiEnv.put(JMSEpr.MAX_SESSIONS_PER_CONNECTION, Integer.toString(MAX_SESSIONS_PER_CONN));

        MockJndiContextHandler.objects.put("ConnectionFactory", new MockJMSConnectionFactory(MAX_SESSIONS_PER_CONN));
        JmsConnectionPool connPool = new JmsConnectionPool(jndiEnv);
        List<JmsConnectionPool.JmsSessionPool> sessPools = connPool.getSessionPools();

        assertEquals(20, connPool.getMaxSessions());
        assertEquals(MAX_SESSIONS_PER_CONN, connPool.getMaxSessionsPerConnection());
        assertEquals(0, sessPools.size());

        try {
            JmsSession session1 = connPool.getSession();

            // Just get 1 session.  Make sure it's returned to the
            // pool on closing...
            assertEquals(1, sessPools.size());
            assertEquals(0, sessPools.get(0).getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(1, sessPools.get(0).getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            connPool.handleCloseSession(session1);
            assertEquals(1, sessPools.get(0).getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(0, sessPools.get(0).getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));

            JmsSession session2 = connPool.getSession();

            // Now try another session.  Should be same session instance as last time.
            // Make sure it's returned to the pool on closing...
            assertTrue(session1 == session2);
            assertEquals(1, sessPools.size());
            assertEquals(0, sessPools.get(0).getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(1, sessPools.get(0).getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            connPool.handleCloseSession(session2);
            assertEquals(1, sessPools.get(0).getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(0, sessPools.get(0).getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));

            // Now try 3 sessions.  Should cause another pool to be created.
            // Make sure it's returned to the pool on closing...
            session1 = connPool.getSession();
            session2 = connPool.getSession();
            JmsSession session3 = connPool.getSession();

            assertEquals(2, sessPools.size());
            assertEquals(0, sessPools.get(0).getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(2, sessPools.get(0).getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(0, sessPools.get(1).getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(1, sessPools.get(1).getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            connPool.handleCloseSession(session1);
            connPool.handleCloseSession(session2);
            connPool.handleCloseSession(session3);
            assertEquals(2, sessPools.get(0).getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(0, sessPools.get(0).getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(1, sessPools.get(1).getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(0, sessPools.get(1).getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));

            List<JmsSession> sessions = getSessions(connPool, 20);
            assertEquals(10, sessPools.size());
            assertAllInUse(sessPools);
            closeAll(sessions, connPool);
            assertAllFree(sessPools);
        } finally {
            connPool.removeSessionPool();
        }
    }

    public void test_XA() throws NamingException, JMSException, ConnectionException, TransactionStrategyException {
        int MAX_SESSIONS_PER_CONN = 3;

        jndiEnv.put(JMSEpr.MAX_SESSIONS_PER_CONNECTION, Integer.toString(MAX_SESSIONS_PER_CONN));
        jndiEnv.put(JMSEpr.MAX_XA_SESSIONS_PER_CONNECTION, "1");

        MockJndiContextHandler.objects.put("ConnectionFactory", new MockJMSXAConnectionFactory(MAX_SESSIONS_PER_CONN));
        JmsConnectionPool connPool = new JmsConnectionPool(jndiEnv);
        List<JmsConnectionPool.JmsSessionPool> sessPools = connPool.getSessionPools();

        assertEquals(20, connPool.getMaxSessions());
        assertEquals(MAX_SESSIONS_PER_CONN, connPool.getMaxSessionsPerConnection());
        assertEquals(1, connPool.getMaxXASessionsPerConnection());
        assertEquals(0, sessPools.size());

        // There's a limit of 1 XA Session per connection.
        // So as sessions get added, a new Connection will be added for each XA Session.  Non XA Sessions
        // will get filled into connections up to the maxSesionsPerConnection, which is 3 for this test.

        MockTransactionStrategy.isActive = true;
        try {
            TransactionStrategy.setTransactionStrategy(new MockTransactionStrategy());

            connPool.getSession();
            assertEquals(1, sessPools.size());

            connPool.getSession();
            assertEquals(2, sessPools.size());

            connPool.getSession();
            assertEquals(3, sessPools.size());

            TransactionStrategy.setTransactionStrategy(new TransactionStrategy.NullTransactionStrategy());
            connPool.getSession();
            connPool.getSession();
            connPool.getSession();
            connPool.getSession();
            connPool.getSession();
            connPool.getSession();
            assertEquals(3, sessPools.size());

            connPool.getSession();
            assertEquals(4, sessPools.size());

            TransactionStrategy.setTransactionStrategy(new MockTransactionStrategy());

            connPool.getSession();
            assertEquals(4, sessPools.size());

            connPool.getSession();
            assertEquals(5, sessPools.size());

            TransactionStrategy.setTransactionStrategy(new TransactionStrategy.NullTransactionStrategy());

            connPool.getSession();
            connPool.getSession();
            connPool.getSession();
            assertEquals(5, sessPools.size());
        } finally {
            connPool.removeSessionPool();
        }
    }

    private void closeAll(List<JmsSession> sessions, JmsConnectionPool connPool) {
        for(JmsSession session : sessions) {
            connPool.handleCloseSession(session);
        }
    }

    private void assertAllInUse(List<JmsConnectionPool.JmsSessionPool> sessPools) {
        for(JmsConnectionPool.JmsSessionPool sessPool : sessPools) {
            assertEquals(0, sessPool.getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(2, sessPool.getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));
        }
    }

    private void assertAllFree(List<JmsConnectionPool.JmsSessionPool> sessPools) {
        for(JmsConnectionPool.JmsSessionPool sessPool : sessPools) {
            assertEquals(2, sessPool.getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
            assertEquals(0, sessPool.getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));
        }
    }

    private List<JmsSession> getSessions(JmsConnectionPool connPool, int numSessions) throws NamingException, JMSException, ConnectionException {
        List<JmsSession> sessions = new ArrayList<JmsSession>();
        for(int i = 0; i < numSessions; i++) {
            sessions.add(connPool.getSession());
        }
        return sessions;
    }
}
