/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

import org.apache.commons.net.ftp.FTP;
import org.apache.commons.net.ftp.FTPClient;
import org.jboss.internal.soa.esb.util.FtpUtils;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.util.RemoteFileSystem;
import org.jboss.soa.esb.util.RemoteFileSystemException;

/**
 * Simplified FTP transfers
 * <p>
 * Description: Implements a simple set of FTP functionality Parameters to
 * establish the FTP connection are provided at construction time and cannot
 * change during the lifetime of the object <br/>Hides low level details.
 * </p>
 * 
 * TODO why duplicate so much of FtpImpl.java?
 */

public class FtpClientUtil
{
        private static final String TMP_SUFFIX = ".rosettaPart";
                
        public enum XFER_TYPE
        {
                ascii, binary
        }

        private ConfigTree m_oParms;

        private String m_sFtpServer, m_sUser, m_sPasswd;

        private String m_sRemoteDir, m_sLocalDir;

        private int m_iPort;

        private boolean m_bPassive;

        private FTPClient m_oConn = new FTPClient();

        private boolean m_bAsciiTransferType ; 
        
        private int renameRetry;

        public String getRemoteDir ()
        {
                return m_sRemoteDir;
        }

        /**
         * Checks validity and completeness of parameters, and keeps the info
         * internally for subsequent FTP requests
         * 
         * @param p_oP
         *            ConfigTree
         * @throws ConfigurationException :
         *             if parameters are invalid or incomplete
         *             <li>Parameters: (XML attributes at the root level) </li>
         *             <li> ftpServer = name or IP of FTP server </li>
         *             <li> ftpUser = login ID for server </li>
         *             <li> ftpPassword </li>
         *             <li> localDirURI = absolute path in the local filesystem
         *             </li>
         *             <li> remoteDirURI = remote path is relative to ftp user home
         *             in remote computer </li>
         * @throws RemoteFileSystemException
         */

        public FtpClientUtil (ConfigTree p_oP, boolean p_bConnect) throws RemoteFileSystemException, ConfigurationException
        {
                m_oParms = p_oP;
                initialize(p_bConnect);
        } // _________________________________

        public FtpClientUtil (List<KeyValuePair> attribs, boolean connect)
                        throws RemoteFileSystemException, ConfigurationException
        {
                m_oParms = new ConfigTree("fromProps");
                for (KeyValuePair oCurr : attribs)
                        m_oParms.setAttribute(oCurr.getKey(), oCurr.getValue());
                initialize(connect);
        } // __________________________________

        private void initialize (boolean bConnect) throws RemoteFileSystemException, ConfigurationException
        {
                checkParms();
                
                try
                {
                        if (bConnect)
                        {
                                if (m_iPort > 0)
                                {
                                    m_oConn.connect(m_sFtpServer, m_iPort) ;
                                }
                                else
                                {
                                    m_oConn.connect(m_sFtpServer) ;
                                }

                                
                                if (!m_oConn.isConnected())
                                        throw new RemoteFileSystemException("Can't connect to FTP server");
                                if (!m_oConn.login(m_sUser, m_sPasswd))
                                {
                                    m_oConn.logout() ;
                                    throw new RemoteFileSystemException("Remote login failed: " + m_oConn.getReplyString());
                                }
                                m_oConn.setFileType(m_bAsciiTransferType ? FTP.ASCII_FILE_TYPE : FTP.BINARY_FILE_TYPE) ;
                                if (m_bPassive)
                                {
                                        m_oConn.enterLocalPassiveMode() ;
                                }
                        }
                }
                catch (IOException ioe)
                {
                        if (m_oConn.isConnected())
                        {
                            try
                            {
                                m_oConn.disconnect() ;
                            }
                            catch (final IOException ioe2) {} // ignore
                        }
                        throw new RemoteFileSystemException(ioe);
                }
        } // __________________________________

        /**
         * Terminates ftp session and frees resources
         * <li>Well behaved programs should make sure to call this method </li>
         */
        public void quit ()
        {
                if (null != m_oConn) try
                {
                        m_oConn.quit();
                        m_oConn.disconnect();
                }
                catch (Exception e)
                {
                }
        } // _________________________________

        /**
         * Deletes specified file in remote directory
         * 
         * @param p_sFile
         *            String : filename to delete. Method will attempt to delete
         *            file with rightmost node of argument within remote directory
         *            specified in 'remoteDirURI'
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or file cannot be
         *             deleted in remote directory
         */
        public void deleteRemoteFile (String p_sFile) throws RemoteFileSystemException
        {
                try
                {
                        changeRemoteDirectory() ;
                        if (!m_oConn.deleteFile(p_sFile))
                        {
                            throw new RemoteFileSystemException("Failed to delete remote file: " + m_oConn.getReplyString());
                        }
                }
                catch (IOException ex)
                {
                        throw new RemoteFileSystemException(ex);
                }
        } // _________________________________

        public void remoteDelete (File p_oFile) throws RemoteFileSystemException
        {
                try
                {
                        changeRemoteDirectory() ;
                        if (!m_oConn.deleteFile(p_oFile.getName()))
                        {
                            throw new RemoteFileSystemException("Failed to delete remote file: " + m_oConn.getReplyString());
                        }
                }
                catch (IOException ex)
                {
                        throw new RemoteFileSystemException(ex);
                }
        } // _________________________________

        /**
         * Gets the list of files in the remote directory that end with arg0
         * 
         * @param p_sSuffix
         *            String : retrieve only files that end with that suffix - all
         *            files if null
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or problems
         *             encountered
         */
        public String[] getFileListFromRemoteDir (String p_sSuffix)
                        throws RemoteFileSystemException
        {
                String sSuffix = (null == p_sSuffix) ? "*" : "*" + p_sSuffix;
                
                try
                {
                        changeRemoteDirectory() ;
                        return m_oConn.listNames(sSuffix);
                }
                catch (IOException ex)
                {
                        throw new RemoteFileSystemException(ex);
                }
        } // _________________________________

        /**
         * Change remote directory
         * 
         * @param p_sDir
         *            String : directory to set
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or problems
         *             encountered
         */
        public void setRemoteDir (String p_sDir) throws RemoteFileSystemException
        {
                m_sRemoteDir = p_sDir ;
        } // _________________________________

        /**
         * Renames specified file in remote directory to specified new name
         * 
         * @param p_sFrom
         *            String : filename to rename
         * @param p_sTo
         *            String : new filename
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or file cannot be
         *             renamed to new name in remote directory
         *             <li>Method will attempt to rename file with rightmost node
         *             of argument within remote directory specified in
         *             'remoteDirURI', to new name inside the SAME remote directory
         */
        public void renameInRemoteDir (String p_sFrom, String p_sTo)
                        throws RemoteFileSystemException
        {
                try
                {
                        changeRemoteDirectory() ;
                        if (!m_oConn.rename(p_sFrom, p_sTo))
                        {
                            throw new RemoteFileSystemException("Failed to rename file: " + m_oConn.getReplyString());
                        }
                }
                catch (final IOException ioe)
                {
                        throw new RemoteFileSystemException(ioe);
                }
        } // _________________________________

        public void remoteRename (File p_oFrom, File p_oTo) throws RemoteFileSystemException
        {
                try
                {
                        if (!m_oConn.rename(FtpClientUtil.fileToFtpString(p_oFrom),
                                FtpUtils.fileToFtpString(p_oTo)))
                        {
                            throw new RemoteFileSystemException("Failed to rename file: " + m_oConn.getReplyString());
                        }
                }
                catch (IOException ioe)
                {
                        throw new RemoteFileSystemException(ioe);
                }
        } // _________________________________

        /**
         * Uploads specified file from local directory (localDirURI) to remote
         * directory (remoteDirURI)
         * 
         * @param p_oFile
         *            String : filename to upload
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or file cannot be
         *             uploaded
         *             <li> local file will be renamed during transfer
         *             ('.xferNotReady' appended to name)</li>
         *             <li> upon successful completion. the suffix '.xferDone' will
         *             be appended to the original filename </li>
         */
        public void uploadFile (File p_oFile, String p_sRemoteName)
                        throws RemoteFileSystemException
        {
                try
                {
                        changeRemoteDirectory() ;
                        final String sRemoteTmp = p_sRemoteName + TMP_SUFFIX ;
                        
                        final OutputStream os = m_oConn.storeFileStream(sRemoteTmp) ;
                        try
                        {
                            final FileInputStream fis = new FileInputStream(p_oFile) ;
                            try
                            {
                                copyStream(fis, os) ;
                            }
                            finally
                            {
                                fis.close() ;
                            }
                        }
                        finally
                        {
                            os.flush() ;
                            os.close() ;
                        }
                        if (!m_oConn.completePendingCommand())
                        {
                            throw new RemoteFileSystemException("Failed to upload contents: " + m_oConn.getReplyString()) ;
                        }
                        
                        boolean retryProblem = true;  // https://jira.jboss.org/jira/browse/JBESB-1995

                        for (int i = 0; i < renameRetry; i++)
                        {
                            if (m_oConn.rename(sRemoteTmp, p_sRemoteName))
                            {
                                retryProblem = false;

                                break;
                            }
                            else
                            {
                                if (i+1 < renameRetry)
                                {
                                    try
                                    {
                                        Thread.sleep(RemoteFileSystem.DEFAULT_RENAME_RETRY_TIMEOUT);
                                    }
                                    catch (final Exception ex)
                                    {
                                    }
                                }
                            }
                        }

                        if (retryProblem)
                            throw new RemoteFileSystemException("Failed to rename file: " + m_oConn.getReplyString());
                }
                catch (IOException ex)
                {
                        throw new RemoteFileSystemException(ex);
                }
        } // _________________________________

        /**
         * Downloads specified file from remote directory (remoteDirURI) to local
         * directory (localDirURI)
         * 
         * @param p_sFile
         *            String : filename to download
         * @throws RemoteFileSystemException :
         *             if ftp connection cannot be established, or file cannot be
         *             downloaded
         *             <li> local file is assigned a temporary name during transfer
         *             </li>
         *             <li> upon successful completion, local temporary file will be
         *             renamed to name specified in argument, and suffix '.xferDone'
         *             will be appended to the original filename in the remote
         *             directory </li>
         */
        public void downloadFile (String p_sFile, String p_sFinalName)
                        throws RemoteFileSystemException
        {
                try
                {
                        final File to = new File(p_sFinalName) ;
                        final File oLocalDir = new File(m_sLocalDir);
                        final File oNew = (to.isAbsolute() ? to : new File(oLocalDir, p_sFinalName)) ;
                        if (oNew.exists())
                            oNew.delete();
                        
                        final File toTmp = new File(p_sFinalName + TMP_SUFFIX) ;
                        final File oNewTmp = (toTmp.isAbsolute() ? toTmp : new File(oLocalDir, p_sFinalName + TMP_SUFFIX)) ;
                        if (oNewTmp.exists())
                            oNewTmp.delete();
                        
                        changeRemoteDirectory() ;
                        final InputStream is = m_oConn.retrieveFileStream(p_sFile) ;
                        if (is == null)
                        {
                            throw new RemoteFileSystemException("Could not download file: " + m_oConn.getReplyString());
                        }
                        
                        try
                        {
                            final FileOutputStream fos = new FileOutputStream(oNewTmp) ;
                            try
                            {
                                copyStream(is, fos) ;
                            }
                            finally
                            {
                                fos.close() ;
                            }
                        }
                        finally
                        {
                            is.close() ;
                        }
                        if (!m_oConn.completePendingCommand())
                        {
                            oNewTmp.delete() ;
                            throw new RemoteFileSystemException("Failed to download contents: " + m_oConn.getReplyString()) ;
                        }
                        FileUtil.renameTo(oNewTmp, oNew);
                }
                catch (IOException ex)
                {
                        throw new RemoteFileSystemException(ex);
                }
        } // _________________________________

        private void checkParms () throws ConfigurationException
        {
                m_sFtpServer = m_oParms.getAttribute(RemoteFileSystem.PARMS_FTP_SERVER);
                if (null == m_sFtpServer)
                        throw new ConfigurationException("No FTP server specified");

                m_sUser = m_oParms.getAttribute(RemoteFileSystem.PARMS_USER);
                if (null == m_sUser)
                        throw new ConfigurationException("No username specified for FTP");

                m_sPasswd = m_oParms.getAttribute(RemoteFileSystem.PARMS_PASSWD);
                if (null == m_sPasswd)
                        throw new ConfigurationException("No password specified for FTP");

                m_sRemoteDir = m_oParms.getAttribute(RemoteFileSystem.PARMS_REMOTE_DIR);
                if (null == m_sRemoteDir) 
                        m_sRemoteDir = "";

                m_sLocalDir = m_oParms.getAttribute(RemoteFileSystem.PARMS_LOCAL_DIR);
                if (null == m_sLocalDir) 
                        m_sLocalDir = ".";

                String sAux = m_oParms.getAttribute(RemoteFileSystem.PARMS_PORT);
                m_iPort = (null == sAux) ? 21 : Integer.parseInt(sAux);

                boolean bAscii = false;
                sAux = m_oParms.getAttribute(RemoteFileSystem.PARMS_ASCII);
                
                if (null != sAux) 
                        bAscii = Boolean.parseBoolean(sAux);
                
                m_bAsciiTransferType = bAscii;

                m_bPassive = false;
                sAux = m_oParms.getAttribute(RemoteFileSystem.PARMS_PASSIVE);
                m_bPassive = (null != sAux) && Boolean.parseBoolean(sAux);
                
                String renameRetryString = ModulePropertyManager.getPropertyManager(
                        ModulePropertyManager.TRANSPORTS_MODULE).getProperty(
                        Environment.FTP_RENAME_RETRY, null);

                if (renameRetryString != null)
                {
                    try
                    {
                        renameRetry = Integer.parseInt(renameRetryString);
                    }
                    catch (Exception ex)
                    {
                        throw new ConfigurationException("Invalid rename retry limit: "+renameRetryString);
                    }
                }
                else
                    renameRetry = RemoteFileSystem.DEFAULT_RENAME_RETRY_NUMBER;
                
                if (renameRetry < 1)
                    renameRetry = 1;
        } // __________________________________

        public static String fileToFtpString (File p_oF)
        {
                return (null == p_oF) ? null : p_oF.toString().replace("\\", "/");
        } // ________________________________
        
        private void changeRemoteDirectory()
            throws IOException, RemoteFileSystemException
        {
            final String remoteDir = getRemoteDir() ;
            if ((remoteDir != null) && (remoteDir.length() > 0))
            {
                if (!m_oConn.changeWorkingDirectory(remoteDir))
                {
                    throw new RemoteFileSystemException("Failed to change to remote directory: " + m_oConn.getReplyString());
                }
            }
        }
        
        private void copyStream(final InputStream is, final OutputStream os)
            throws IOException
        {
            final BufferedInputStream bis = new BufferedInputStream(is) ;
            final BufferedOutputStream bos = new BufferedOutputStream(os) ;
            
            final byte[] buffer = new byte[256] ;
            while(true)
            {
                final int count = bis.read(buffer) ;
                if (count <= 0)
                {
                    break ;
                }
                bos.write(buffer, 0, count) ;
            }
            bos.flush() ;
        }
} // ____________________________________________________________________________
