/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security.util;

import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;

import org.jboss.soa.esb.services.security.SecurityServiceException;

/**
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public final class CryptoUtil
{
    private CryptoUtil() {}
    
    public static Cipher getCipher(final String transformation) throws SecurityServiceException
    {
        try
        {
            return Cipher.getInstance(transformation);
        }
        catch (NoSuchAlgorithmException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (NoSuchPaddingException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
    }
    
    public static Cipher getCipherForDecryption(final String transformation, final Key key) throws SecurityServiceException
    {
        final Cipher cipher = getCipher(transformation);
        try
        {
            cipher.init(Cipher.DECRYPT_MODE, key);
        }
        catch (InvalidKeyException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        return cipher;
    }
    
    public static Cipher getCipherForEncryption(final String transformation, final Key key) throws SecurityServiceException
    {
        final Cipher cipher = getCipher(transformation);
        try
        {
            cipher.init(Cipher.ENCRYPT_MODE, key);
        }
        catch (InvalidKeyException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        return cipher;
    }
    

    public static byte[] decrypt(final byte[] bytes, final Key key, final String transformation) throws SecurityServiceException
    {
        try
        {
            Cipher cipher = getCipher(transformation);
            cipher.init(Cipher.DECRYPT_MODE, key);
            return cipher.doFinal(bytes);
        }
        catch (final InvalidKeyException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (final IllegalBlockSizeException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (final BadPaddingException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
    }

    public static byte[] encrypt(byte[] text, Key key, String transformation) throws SecurityServiceException
    {
        try
        {
            Cipher cipher = getCipher(transformation);
            cipher.init(Cipher.ENCRYPT_MODE, key);
            return cipher.doFinal(text);
        }
        catch (InvalidKeyException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (IllegalBlockSizeException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (BadPaddingException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
    }

    public static byte[] copyBytes(final byte[] bytes, final int length)
    {
        byte[] newArr = null;
        if (bytes.length == length)
        {
            newArr = bytes;
        } else
        {
            newArr = new byte[length];
            for (int i = 0; i < length; i++)
            {
                newArr[i] = bytes[i];
            }
        }
        return newArr;
    }

}
