/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security.auth;

import java.io.Serializable;
import java.util.Set;

import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.security.PublicCryptoUtil;
import org.jboss.soa.esb.services.security.SecurityService;
import org.jboss.soa.esb.services.security.SecurityServiceException;

/**
 * Util class for operations common to security extractors.
 * <p/>
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * 
 */
public final class ExtractorUtil
{
    private ExtractorUtil()
    {
    }

    /**
     * This method will try to extract security information from the passed-in payload of type T.
     * The extractors will iterated over and given the chance to extract, the first one that is successful
     * will have the {@link AuthenticationRequest} that it populated returned.
     * 
     * @param <T> The type of the payload.
     * @param payload The payload that will be of type T
     * @param extractors The extractors to extract security information from.
     * @return {@link AuthenticationRequest} A populated AuthenticationReqeust or null if no security information was extracted.
     * @throws ExtractionException
     */
    public static <T> AuthenticationRequest extract(final T payload, final Set<SecurityInfoExtractor<T>> extractors) throws ExtractionException
    {
        for (SecurityInfoExtractor<T> extractor : extractors)
        {
            final AuthenticationRequest authRequest = extractor.extractSecurityInfo(payload);
            if (authRequest != null)
            {
                return authRequest;
            }
        }
        
        return null;
    }

    /**
     * This method will add the passed-in {@link AuthenticationRequest} to the Message context.
     * 
     * @param authRequest The AuthenticationRequest to add to the Message context
     * @param message The ESB Message object instance.
     * @throws SecurityServiceException
     */
    public static void addAuthRequestToMessage(final AuthenticationRequest authRequest, final Message message) throws SecurityServiceException
    {
        if (authRequest != null)
        {
            final byte[] encrypted = PublicCryptoUtil.INSTANCE.encrypt((Serializable) authRequest);
            if (encrypted != null)
            {
                message.getContext().setContext(SecurityService.AUTH_REQUEST, encrypted);
            }
            else
            {
                throw new SecurityServiceException("No public keystore has been configured which means that the authentication request cannot be encrypted. Please configure jbossesb-properties.xml with a publickey store.");
            }
        }
    }

}
