package org.jboss.soa.esb.message.body.content;

import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * Different message formats represent views on the underlying Message Body,
 * i.e., payload. They do not change how the content is represented on the wire,
 * only how it can be more easily interpreted within the application. These are
 * essentially helper classes.
 * 
 * In order to use, first create your Message (via the factory) and then pass it
 * to the appropriate method of this class. Use the returned Body type to then
 * populate your Message before sending it. At the receiver, you can get the type by
 * calling bodyType and then use the right method to get the Body type back for
 * use. If you try to use it as the wrong type, you'll get an exception.
 * 
 * It is not possible currently to change between types, i.e., to give different
 * "views" on to the same underlying message structure. However, that is relatively
 * simple to implement if needed.
 */

public class Payload
{
	public static final String CONTENT_TYPE = "org.jboss.soa.esb.message.body.content";
	
	public static final String RAW_BODY = "raw";
	public static final String BYTES_BODY = "bytes";
	public static final String MAP_BODY = "map";
	public static final String OBJECT_BODY = "object";
	public static final String TEXT_BODY = "text";
	
	public static final String bodyType (Message msg)
	{
		if ((msg == null) || (msg.getBody() == null))
			throw new IllegalArgumentException();
		
		Body body = msg.getBody();
		
		String type = (String) body.get(CONTENT_TYPE);
		
		if (type == null)
			return RAW_BODY;
		else
			return type;
	}
	
	public static final BytesBody createBytesBody (Message msg) throws InvalidPayloadException
	{
		if ((msg == null) || (msg.getBody() == null))
			return null;
		
		checkType(msg, BYTES_BODY);
		
		return (BytesBody) MessageFactory.getInstance().createBodyType(msg, BYTES_BODY);
	}

	public static final MapBody createMapBody (Message msg) throws InvalidPayloadException
	{
		if ((msg == null) || (msg.getBody() == null))
			return null;
		
		checkType(msg, MAP_BODY);
		
		return (MapBody) MessageFactory.getInstance().createBodyType(msg, MAP_BODY);
	}
	
	public static final ObjectBody createObjectBody (Message msg) throws InvalidPayloadException
	{
		if ((msg == null) || (msg.getBody() == null))
			return null;
		
		checkType(msg, OBJECT_BODY);
		
		return (ObjectBody) MessageFactory.getInstance().createBodyType(msg, OBJECT_BODY);
	}
	
	public static final TextBody createTextBody (Message msg) throws InvalidPayloadException
	{
		if ((msg == null) || (msg.getBody() == null))
			return null;
		
		checkType(msg, TEXT_BODY);
		
		return (TextBody) MessageFactory.getInstance().createBodyType(msg, TEXT_BODY);
	}
	
	/*
	 * If the message type is set and it does not conform the type expected,
	 * then throw IllegalArgumentException.
	 */
	
	private static final void checkType (Message msg, String required) throws InvalidPayloadException
	{
		String type = (String) msg.getBody().get(CONTENT_TYPE);
		
		if ((type != null) && (!type.equals(required)))
			throw new InvalidPayloadException();
	}
}