/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.message;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.actions.ActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;

/**
 * Instance of ActionPipelineProcessor to allow selective overriding of the methods
 * on a single instance of the action processor.
 * @author kevin
 */
final class OverriddenActionPipelineProcessor implements ActionPipelineProcessor
{
    /**
     * The method info.
     */
    private final ActionProcessorMethodInfo methodInfo ;
    /**
     * The processor.
     */
    private final ActionPipelineProcessor processor ;
    
    /**
     * Construct the overridden action pipeline processor.
     * @param config The action config.
     * @param processor The processor.
     * @throws ConfigurationException for errors during configuration.
     */
    OverriddenActionPipelineProcessor(final ConfigTree config, final ActionPipelineProcessor processor)
        throws ConfigurationException
    {
        this.processor = processor ;
        methodInfo = ActionProcessorMethodInfo.getMethodInfo(config, processor.getClass()) ;
    }
    
    /**
     * Perform the action processing on the specified message.
     * This method is invoked for each message passing throught the pipeline.
     * 
     * @param message The current message being processed.
     * @return The message to be passed to the next stage of the pipeline.
     * @throws ActionProcessingException for errors during processing.
     */
    public Message process(final Message message)
        throws ActionProcessingException
    {
        if (methodInfo.hasProcessMethods())
        {
            return methodInfo.processMethods(processor, message) ;
        }
        else
        {
            return processor.process(message) ;
        }
    }
    
    /**
     * Process an exception generated by the pipeline processing.
     * Invoked when the processing of a subsequent stage of the
     * pipeline generates an exception.
     * 
     * @param message The original message.
     * @param th The throwable raised by the pipeline processing
     */
    public void processException(final Message message, final Throwable th)
    {
        if (methodInfo.hasProcessException())
        {
            methodInfo.processException(processor, message, th) ;
        }
        else
        {
            processor.processException(message, th) ;
        }
    }
    
    /**
     * Process a successful pipeline process. 
     * Invoked when the pipeline processing completes successfully.
     * 
     * @param message The original message.
     */
    public void processSuccess(final Message message)
    {
        if (methodInfo.hasProcessSuccess())
        {
            methodInfo.processSuccess(processor, message) ;
        }
        else
        {
            processor.processSuccess(message) ;
        }
    }
    
    /**
     * Initialise the action instance.
     * <p/>
     * This method is called after the action instance has been instantiated so that
     * configuration options can be validated.
     * 
     * @throws ActionLifecycleException for errors during initialisation.
     */
    public void initialise()
        throws ActionLifecycleException
    {
        processor.initialise() ;
    }

    /**
     * Destroy the action instance.
     * <p/>
     * This method is called prior to the release of the action instance.  All
     * resources associated with this action instance should be released as the
     * instance will no longer be used.
     */
    public void destroy()
        throws ActionLifecycleException
    {
        processor.destroy() ;
    }
}